"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.createAndStartDestroyService = exports.createAndStartDeployService = exports.terraformPtyService = exports.deployMachine = exports.handleLineReceived = exports.bufferUnterminatedLines = exports.extractVariableNameFromPrompt = exports.isDeployEvent = void 0;
const xstate_1 = require("xstate");
const commons_1 = require("@cdktf/commons");
const errors_1 = require("../errors");
const strip_ansi_1 = __importDefault(require("strip-ansi"));
const os_1 = require("os");
const pty_process_1 = require("./pty-process");
function isDeployEvent(event, type) {
    return event.type === type;
}
exports.isDeployEvent = isDeployEvent;
function extractVariableNameFromPrompt(line) {
    const noColorLine = (0, strip_ansi_1.default)(line);
    const lines = noColorLine.split("\n");
    const lineWithVar = lines.find((line) => line.includes("var."));
    if (!lineWithVar) {
        throw commons_1.Errors.Internal(`Could not find variable name in prompt: ${line}. This is most likely a bug in cdktf. Please report it at https://cdk.tf/bug`);
    }
    return lineWithVar.split("var.")[1].trim();
}
exports.extractVariableNameFromPrompt = extractVariableNameFromPrompt;
// used to only send completed lines (= with a newline at the end of them) to our logic (#2827)
function bufferUnterminatedLines(handler) {
    let buffer = "";
    function bufferedReceiverFunction(output) {
        buffer += output;
        const lines = buffer.split(os_1.EOL);
        // if the string ends with \n this will be an empty string
        // else it will contain an "unfinished" line
        // the fallback to an empty string is to make TS happy and should never happen
        buffer = lines.pop() || "";
        if (lines.length > 0)
            handler(lines.join(os_1.EOL) + "\n");
    }
    bufferedReceiverFunction.getBuffer = () => buffer;
    return bufferedReceiverFunction;
}
exports.bufferUnterminatedLines = bufferUnterminatedLines;
function handleLineReceived(send) {
    return (output) => {
        let hideOutput = false;
        const noColorLine = (0, strip_ansi_1.default)(output);
        // possible events based on line
        if (noColorLine.includes("approved using the UI or API")) {
            send({ type: "APPROVED_EXTERNALLY" });
        }
        else if (noColorLine.includes("discarded using the UI or API")) {
            send({ type: "REJECTED_EXTERNALLY" });
        }
        else if (noColorLine.includes("Do you want to perform these actions") ||
            noColorLine.includes("Do you really want to destroy all resources?") ||
            noColorLine.includes("Do you really want to destroy all resources in workspace")) {
            hideOutput = true;
            send({ type: "OUTPUT_RECEIVED", output });
            send({ type: "REQUEST_APPROVAL" });
        }
        else if (noColorLine.includes("var.") &&
            noColorLine.includes("Enter a value:")) {
            hideOutput = true;
            const variableName = extractVariableNameFromPrompt(output);
            send({
                type: "OUTPUT_RECEIVED",
                output: (0, errors_1.missingVariable)(variableName),
            });
            send({ type: "VARIABLE_MISSING", variableName });
        }
        else if (noColorLine.includes("Do you want to override the soft failed policy check?")) {
            hideOutput = true;
            send({ type: "OUTPUT_RECEIVED", output });
            send({ type: "REQUEST_SENTINEL_OVERRIDE" });
        }
        else if (noColorLine.includes("overridden using the UI or API")) {
            send({ type: "OVERRIDDEN_EXTERNALLY" });
        }
        if (!hideOutput) {
            send({
                type: "OUTPUT_RECEIVED",
                output,
            });
        }
    };
}
exports.handleLineReceived = handleLineReceived;
exports.deployMachine = (0, xstate_1.createMachine)({
    predictableActionArguments: true,
    context: {},
    initial: "idle",
    id: "root",
    states: {
        idle: {
            on: {
                START: { target: "running" },
            },
        },
        running: {
            invoke: {
                id: "pty",
                src: "runTerraformInPty",
            },
            on: {
                EXITED: "exited",
                STOP: "stopped",
            },
            initial: "processing",
            states: {
                // TODO: what else might TF CLI be asking? Can we detect any question from the TF CLI to show a good error?
                processing: {
                    on: {
                        REQUEST_APPROVAL: "awaiting_approval",
                        REQUEST_SENTINEL_OVERRIDE: "awaiting_sentinel_override",
                        VARIABLE_MISSING: {
                            actions: (0, xstate_1.send)({ type: "EXITED", exitCode: 1 }),
                        },
                    },
                },
                awaiting_approval: {
                    on: {
                        APPROVED_EXTERNALLY: "processing",
                        REJECTED_EXTERNALLY: {
                            target: "#root.exited",
                            actions: (0, xstate_1.assign)({ cancelled: true }),
                        },
                        APPROVE: {
                            target: "processing",
                            actions: (0, xstate_1.send)({ type: "SEND_LINE", input: "yes" }, { to: "pty" }),
                        },
                        REJECT: {
                            target: "processing",
                            actions: [
                                (0, xstate_1.send)({ type: "SEND_LINE", input: "no" }, { to: "pty" }),
                                (0, xstate_1.assign)({
                                    cancelled: true,
                                }),
                            ],
                        },
                    },
                },
                awaiting_sentinel_override: {
                    on: {
                        OVERRIDDEN_EXTERNALLY: "processing",
                        OVERRIDE_REJECTED_EXTERNALLY: {
                            target: "#root.exited",
                            actions: (0, xstate_1.assign)({ cancelled: true }),
                        },
                        // This is a bit of a hack, because the external discard message
                        // posted by Terraform UI is the same as during apply. So, we capture that
                        // and emit our own event to make it more specific.
                        REJECTED_EXTERNALLY: {
                            actions: (0, xstate_1.send)({ type: "OVERRIDE_REJECTED_EXTERNALLY" }),
                        },
                        OVERRIDE: {
                            target: "processing",
                            actions: (0, xstate_1.send)({ type: "SEND_LINE", input: "override" }, { to: "pty" }),
                        },
                        REJECT_OVERRIDE: {
                            target: "processing",
                            actions: [
                                (0, xstate_1.send)({ type: "SEND_LINE", input: "no" }, { to: "pty" }),
                                (0, xstate_1.assign)({
                                    cancelled: true,
                                }),
                            ],
                        },
                    },
                },
            },
        },
        exited: { type: "final" },
        stopped: { type: "final" },
    },
}, {
    services: {
        runTerraformInPty: (context, event) => terraformPtyService(context, event, pty_process_1.spawnPty),
    },
});
function terraformPtyService(_context, event, spawn = pty_process_1.spawnPty) {
    return (send, onReceive) => {
        if (event.type !== "START") {
            throw commons_1.Errors.Internal(`Terraform CLI invocation state machine: Unexpected event caused transition to the running state: ${event.type}`);
        }
        // Communication from the pty to the caller
        const receiver = bufferUnterminatedLines(handleLineReceived(send));
        const { exitCode, actions } = spawn(event.pty, (data) => {
            receiver(data);
        });
        // Communication from the caller to the pty
        onReceive((event) => {
            if (event.type === "SEND_LINE") {
                actions.writeLine(event.input);
            }
        });
        exitCode.then((exitCode) => {
            const lastBuffer = receiver.getBuffer();
            if (lastBuffer.length > 0) {
                commons_1.logger.debug(`Terraform CLI exited but the last outputted line was not terminated with a newline and hence is still in the buffer and wasn't printed: "${lastBuffer}"`);
            }
            send({ type: "EXITED", exitCode });
        });
        return () => {
            commons_1.logger.trace("Terraform CLI state machine: cleaning up pty");
            actions.stop();
        };
    };
}
exports.terraformPtyService = terraformPtyService;
function createAndStartDeployService(options) {
    var _a, _b;
    const service = (0, xstate_1.interpret)(exports.deployMachine);
    const args = [
        "apply",
        ...(options.autoApprove ? ["-auto-approve"] : []),
        // "-input=false", we can't use this anymore but TODO: we need to detect TF CLI asking for missing inputs and either allow passing them or stop there and fail
        ...options.extraOptions,
        ...(options.refreshOnly ? ["-refresh-only"] : []),
        ...(options.noColor ? ["-no-color"] : []),
        ...(options.parallelism > -1
            ? [`-parallelism=${options.parallelism}`]
            : []),
    ];
    (_a = options.vars) === null || _a === void 0 ? void 0 : _a.forEach((v) => {
        args.push(`-var=${v}`);
    });
    (_b = options.varFiles) === null || _b === void 0 ? void 0 : _b.forEach((v) => {
        args.push(`-var-file=${v}`);
    });
    commons_1.logger.debug(`Executing ${options.terraformBinaryName} ${args.join(" ")} in ${options.workdir}`);
    const config = {
        file: options.terraformBinaryName,
        args,
        options: {
            cwd: options.workdir,
            env: process.env, // TODO: make this explicit and move to caller or whatever
        },
    };
    service.send({ type: "START", pty: config });
    return service;
}
exports.createAndStartDeployService = createAndStartDeployService;
function createAndStartDestroyService(options) {
    var _a, _b;
    const service = (0, xstate_1.interpret)(exports.deployMachine);
    const args = [
        "destroy",
        ...(options.autoApprove ? ["-auto-approve"] : []),
        // "-input=false", we can't use this anymore but TODO: we need to detect TF CLI asking for missing inputs and either allow passing them or stop there and fail
        ...options.extraOptions,
        ...(options.noColor ? ["-no-color"] : []),
        ...(options.parallelism > -1
            ? [`-parallelism=${options.parallelism}`]
            : []),
    ];
    (_a = options.vars) === null || _a === void 0 ? void 0 : _a.forEach((v) => {
        args.push(`-var=${v}`);
    });
    (_b = options.varFiles) === null || _b === void 0 ? void 0 : _b.forEach((v) => {
        args.push(`-var-file=${v}`);
    });
    commons_1.logger.debug(`Executing ${options.terraformBinaryName} ${args.join(" ")} in ${options.workdir}`);
    const config = {
        file: options.terraformBinaryName,
        args,
        options: {
            cwd: options.workdir,
            env: process.env, // TODO: make this explicit and move to caller or whatever
        },
    };
    service.send({ type: "START", pty: config });
    return service;
}
exports.createAndStartDestroyService = createAndStartDestroyService;
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoiZGVwbG95LW1hY2hpbmUuanMiLCJzb3VyY2VSb290IjoiIiwic291cmNlcyI6WyJkZXBsb3ktbWFjaGluZS50cyJdLCJuYW1lcyI6W10sIm1hcHBpbmdzIjoiO0FBQUE7OztHQUdHOzs7Ozs7QUFFSCxtQ0FRZ0I7QUFFaEIsNENBQWdEO0FBQ2hELHNDQUE0QztBQUM1Qyw0REFBbUM7QUFDbkMsMkJBQXlCO0FBQ3pCLCtDQUF5QztBQWtDekMsU0FBZ0IsYUFBYSxDQUMzQixLQUFrQixFQUNsQixJQUFxQjtJQUVyQixPQUFPLEtBQUssQ0FBQyxJQUFJLEtBQUssSUFBSSxDQUFDO0FBQzdCLENBQUM7QUFMRCxzQ0FLQztBQWdDRCxTQUFnQiw2QkFBNkIsQ0FBQyxJQUFZO0lBQ3hELE1BQU0sV0FBVyxHQUFHLElBQUEsb0JBQVMsRUFBQyxJQUFJLENBQUMsQ0FBQztJQUNwQyxNQUFNLEtBQUssR0FBRyxXQUFXLENBQUMsS0FBSyxDQUFDLElBQUksQ0FBQyxDQUFDO0lBQ3RDLE1BQU0sV0FBVyxHQUFHLEtBQUssQ0FBQyxJQUFJLENBQUMsQ0FBQyxJQUFJLEVBQUUsRUFBRSxDQUFDLElBQUksQ0FBQyxRQUFRLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQztJQUNoRSxJQUFJLENBQUMsV0FBVyxFQUFFO1FBQ2hCLE1BQU0sZ0JBQU0sQ0FBQyxRQUFRLENBQ25CLDJDQUEyQyxJQUFJLDhFQUE4RSxDQUM5SCxDQUFDO0tBQ0g7SUFDRCxPQUFPLFdBQVcsQ0FBQyxLQUFLLENBQUMsTUFBTSxDQUFDLENBQUMsQ0FBQyxDQUFDLENBQUMsSUFBSSxFQUFFLENBQUM7QUFDN0MsQ0FBQztBQVZELHNFQVVDO0FBV0QsK0ZBQStGO0FBQy9GLFNBQWdCLHVCQUF1QixDQUNyQyxPQUFpQztJQUVqQyxJQUFJLE1BQU0sR0FBRyxFQUFFLENBQUM7SUFDaEIsU0FBUyx3QkFBd0IsQ0FBQyxNQUFjO1FBQzlDLE1BQU0sSUFBSSxNQUFNLENBQUM7UUFDakIsTUFBTSxLQUFLLEdBQUcsTUFBTSxDQUFDLEtBQUssQ0FBQyxRQUFHLENBQUMsQ0FBQztRQUVoQywwREFBMEQ7UUFDMUQsNENBQTRDO1FBQzVDLDhFQUE4RTtRQUM5RSxNQUFNLEdBQUcsS0FBSyxDQUFDLEdBQUcsRUFBRSxJQUFJLEVBQUUsQ0FBQztRQUUzQixJQUFJLEtBQUssQ0FBQyxNQUFNLEdBQUcsQ0FBQztZQUFFLE9BQU8sQ0FBQyxLQUFLLENBQUMsSUFBSSxDQUFDLFFBQUcsQ0FBQyxHQUFHLElBQUksQ0FBQyxDQUFDO0lBQ3hELENBQUM7SUFDRCx3QkFBd0IsQ0FBQyxTQUFTLEdBQUcsR0FBRyxFQUFFLENBQUMsTUFBTSxDQUFDO0lBRWxELE9BQU8sd0JBQXdCLENBQUM7QUFDbEMsQ0FBQztBQWxCRCwwREFrQkM7QUFFRCxTQUFnQixrQkFBa0IsQ0FBQyxJQUFrQztJQUNuRSxPQUFPLENBQUMsTUFBYyxFQUFFLEVBQUU7UUFDeEIsSUFBSSxVQUFVLEdBQUcsS0FBSyxDQUFDO1FBQ3ZCLE1BQU0sV0FBVyxHQUFHLElBQUEsb0JBQVMsRUFBQyxNQUFNLENBQUMsQ0FBQztRQUV0QyxnQ0FBZ0M7UUFDaEMsSUFBSSxXQUFXLENBQUMsUUFBUSxDQUFDLDhCQUE4QixDQUFDLEVBQUU7WUFDeEQsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLHFCQUFxQixFQUFFLENBQUMsQ0FBQztTQUN2QzthQUFNLElBQUksV0FBVyxDQUFDLFFBQVEsQ0FBQywrQkFBK0IsQ0FBQyxFQUFFO1lBQ2hFLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxxQkFBcUIsRUFBRSxDQUFDLENBQUM7U0FDdkM7YUFBTSxJQUNMLFdBQVcsQ0FBQyxRQUFRLENBQUMsc0NBQXNDLENBQUM7WUFDNUQsV0FBVyxDQUFDLFFBQVEsQ0FBQyw4Q0FBOEMsQ0FBQztZQUNwRSxXQUFXLENBQUMsUUFBUSxDQUNsQiwwREFBMEQsQ0FDM0QsRUFDRDtZQUNBLFVBQVUsR0FBRyxJQUFJLENBQUM7WUFDbEIsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFDMUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLGtCQUFrQixFQUFFLENBQUMsQ0FBQztTQUNwQzthQUFNLElBQ0wsV0FBVyxDQUFDLFFBQVEsQ0FBQyxNQUFNLENBQUM7WUFDNUIsV0FBVyxDQUFDLFFBQVEsQ0FBQyxnQkFBZ0IsQ0FBQyxFQUN0QztZQUNBLFVBQVUsR0FBRyxJQUFJLENBQUM7WUFFbEIsTUFBTSxZQUFZLEdBQUcsNkJBQTZCLENBQUMsTUFBTSxDQUFDLENBQUM7WUFDM0QsSUFBSSxDQUFDO2dCQUNILElBQUksRUFBRSxpQkFBaUI7Z0JBQ3ZCLE1BQU0sRUFBRSxJQUFBLHdCQUFlLEVBQUMsWUFBWSxDQUFDO2FBQ3RDLENBQUMsQ0FBQztZQUNILElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxrQkFBa0IsRUFBRSxZQUFZLEVBQUUsQ0FBQyxDQUFDO1NBQ2xEO2FBQU0sSUFDTCxXQUFXLENBQUMsUUFBUSxDQUNsQix1REFBdUQsQ0FDeEQsRUFDRDtZQUNBLFVBQVUsR0FBRyxJQUFJLENBQUM7WUFDbEIsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLGlCQUFpQixFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7WUFFMUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLDJCQUEyQixFQUFFLENBQUMsQ0FBQztTQUM3QzthQUFNLElBQUksV0FBVyxDQUFDLFFBQVEsQ0FBQyxnQ0FBZ0MsQ0FBQyxFQUFFO1lBQ2pFLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSx1QkFBdUIsRUFBRSxDQUFDLENBQUM7U0FDekM7UUFFRCxJQUFJLENBQUMsVUFBVSxFQUFFO1lBQ2YsSUFBSSxDQUFDO2dCQUNILElBQUksRUFBRSxpQkFBaUI7Z0JBQ3ZCLE1BQU07YUFDUCxDQUFDLENBQUM7U0FDSjtJQUNILENBQUMsQ0FBQztBQUNKLENBQUM7QUFwREQsZ0RBb0RDO0FBRVksUUFBQSxhQUFhLEdBQUcsSUFBQSxzQkFBYSxFQUt4QztJQUNFLDBCQUEwQixFQUFFLElBQUk7SUFDaEMsT0FBTyxFQUFFLEVBQUU7SUFDWCxPQUFPLEVBQUUsTUFBTTtJQUNmLEVBQUUsRUFBRSxNQUFNO0lBQ1YsTUFBTSxFQUFFO1FBQ04sSUFBSSxFQUFFO1lBQ0osRUFBRSxFQUFFO2dCQUNGLEtBQUssRUFBRSxFQUFFLE1BQU0sRUFBRSxTQUFTLEVBQUU7YUFDN0I7U0FDRjtRQUNELE9BQU8sRUFBRTtZQUNQLE1BQU0sRUFBRTtnQkFDTixFQUFFLEVBQUUsS0FBSztnQkFDVCxHQUFHLEVBQUUsbUJBQW1CO2FBQ3pCO1lBQ0QsRUFBRSxFQUFFO2dCQUNGLE1BQU0sRUFBRSxRQUFRO2dCQUNoQixJQUFJLEVBQUUsU0FBUzthQUNoQjtZQUNELE9BQU8sRUFBRSxZQUFZO1lBQ3JCLE1BQU0sRUFBRTtnQkFDTiwyR0FBMkc7Z0JBQzNHLFVBQVUsRUFBRTtvQkFDVixFQUFFLEVBQUU7d0JBQ0YsZ0JBQWdCLEVBQUUsbUJBQW1CO3dCQUNyQyx5QkFBeUIsRUFBRSw0QkFBNEI7d0JBQ3ZELGdCQUFnQixFQUFFOzRCQUNoQixPQUFPLEVBQUUsSUFBQSxhQUFJLEVBQUMsRUFBRSxJQUFJLEVBQUUsUUFBUSxFQUFFLFFBQVEsRUFBRSxDQUFDLEVBQUUsQ0FBQzt5QkFDL0M7cUJBQ0Y7aUJBQ0Y7Z0JBQ0QsaUJBQWlCLEVBQUU7b0JBQ2pCLEVBQUUsRUFBRTt3QkFDRixtQkFBbUIsRUFBRSxZQUFZO3dCQUNqQyxtQkFBbUIsRUFBRTs0QkFDbkIsTUFBTSxFQUFFLGNBQWM7NEJBQ3RCLE9BQU8sRUFBRSxJQUFBLGVBQU0sRUFHYixFQUFFLFNBQVMsRUFBRSxJQUFJLEVBQUUsQ0FBQzt5QkFDdkI7d0JBQ0QsT0FBTyxFQUFFOzRCQUNQLE1BQU0sRUFBRSxZQUFZOzRCQUNwQixPQUFPLEVBQUUsSUFBQSxhQUFJLEVBQ1gsRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxLQUFLLEVBQUUsRUFDbkMsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLENBQ2Q7eUJBQ0Y7d0JBQ0QsTUFBTSxFQUFFOzRCQUNOLE1BQU0sRUFBRSxZQUFZOzRCQUNwQixPQUFPLEVBQUU7Z0NBQ1AsSUFBQSxhQUFJLEVBQUMsRUFBRSxJQUFJLEVBQUUsV0FBVyxFQUFFLEtBQUssRUFBRSxJQUFJLEVBQUUsRUFBRSxFQUFFLEVBQUUsRUFBRSxLQUFLLEVBQUUsQ0FBQztnQ0FDdkQsSUFBQSxlQUFNLEVBQWtEO29DQUN0RCxTQUFTLEVBQUUsSUFBSTtpQ0FDaEIsQ0FBQzs2QkFDSDt5QkFDRjtxQkFDRjtpQkFDRjtnQkFDRCwwQkFBMEIsRUFBRTtvQkFDMUIsRUFBRSxFQUFFO3dCQUNGLHFCQUFxQixFQUFFLFlBQVk7d0JBQ25DLDRCQUE0QixFQUFFOzRCQUM1QixNQUFNLEVBQUUsY0FBYzs0QkFDdEIsT0FBTyxFQUFFLElBQUEsZUFBTSxFQUdiLEVBQUUsU0FBUyxFQUFFLElBQUksRUFBRSxDQUFDO3lCQUN2Qjt3QkFDRCxnRUFBZ0U7d0JBQ2hFLDBFQUEwRTt3QkFDMUUsbURBQW1EO3dCQUNuRCxtQkFBbUIsRUFBRTs0QkFDbkIsT0FBTyxFQUFFLElBQUEsYUFBSSxFQUFDLEVBQUUsSUFBSSxFQUFFLDhCQUE4QixFQUFFLENBQUM7eUJBQ3hEO3dCQUNELFFBQVEsRUFBRTs0QkFDUixNQUFNLEVBQUUsWUFBWTs0QkFDcEIsT0FBTyxFQUFFLElBQUEsYUFBSSxFQUNYLEVBQUUsSUFBSSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsVUFBVSxFQUFFLEVBQ3hDLEVBQUUsRUFBRSxFQUFFLEtBQUssRUFBRSxDQUNkO3lCQUNGO3dCQUNELGVBQWUsRUFBRTs0QkFDZixNQUFNLEVBQUUsWUFBWTs0QkFDcEIsT0FBTyxFQUFFO2dDQUNQLElBQUEsYUFBSSxFQUFDLEVBQUUsSUFBSSxFQUFFLFdBQVcsRUFBRSxLQUFLLEVBQUUsSUFBSSxFQUFFLEVBQUUsRUFBRSxFQUFFLEVBQUUsS0FBSyxFQUFFLENBQUM7Z0NBQ3ZELElBQUEsZUFBTSxFQUdKO29DQUNBLFNBQVMsRUFBRSxJQUFJO2lDQUNoQixDQUFDOzZCQUNIO3lCQUNGO3FCQUNGO2lCQUNGO2FBQ0Y7U0FDRjtRQUNELE1BQU0sRUFBRSxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUU7UUFDekIsT0FBTyxFQUFFLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRTtLQUMzQjtDQUNGLEVBQ0Q7SUFDRSxRQUFRLEVBQUU7UUFDUixpQkFBaUIsRUFBRSxDQUFDLE9BQU8sRUFBRSxLQUFLLEVBQUUsRUFBRSxDQUNwQyxtQkFBbUIsQ0FBQyxPQUFPLEVBQUUsS0FBSyxFQUFFLHNCQUFRLENBQUM7S0FDaEQ7Q0FDRixDQUNGLENBQUM7QUFFRixTQUFnQixtQkFBbUIsQ0FDakMsUUFBdUIsRUFDdkIsS0FBa0IsRUFDbEIsS0FBSyxHQUFHLHNCQUFRO0lBRWhCLE9BQU8sQ0FBQyxJQUF5QixFQUFFLFNBQWdDLEVBQUUsRUFBRTtRQUNyRSxJQUFJLEtBQUssQ0FBQyxJQUFJLEtBQUssT0FBTyxFQUFFO1lBQzFCLE1BQU0sZ0JBQU0sQ0FBQyxRQUFRLENBQ25CLG9HQUFvRyxLQUFLLENBQUMsSUFBSSxFQUFFLENBQ2pILENBQUM7U0FDSDtRQUVELDJDQUEyQztRQUMzQyxNQUFNLFFBQVEsR0FBRyx1QkFBdUIsQ0FBQyxrQkFBa0IsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFDO1FBQ25FLE1BQU0sRUFBRSxRQUFRLEVBQUUsT0FBTyxFQUFFLEdBQUcsS0FBSyxDQUFDLEtBQUssQ0FBQyxHQUFHLEVBQUUsQ0FBQyxJQUFJLEVBQUUsRUFBRTtZQUN0RCxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUM7UUFDakIsQ0FBQyxDQUFDLENBQUM7UUFFSCwyQ0FBMkM7UUFDM0MsU0FBUyxDQUFDLENBQUMsS0FBa0IsRUFBRSxFQUFFO1lBQy9CLElBQUksS0FBSyxDQUFDLElBQUksS0FBSyxXQUFXLEVBQUU7Z0JBQzlCLE9BQU8sQ0FBQyxTQUFTLENBQUMsS0FBSyxDQUFDLEtBQUssQ0FBQyxDQUFDO2FBQ2hDO1FBQ0gsQ0FBQyxDQUFDLENBQUM7UUFFSCxRQUFRLENBQUMsSUFBSSxDQUFDLENBQUMsUUFBUSxFQUFFLEVBQUU7WUFDekIsTUFBTSxVQUFVLEdBQUcsUUFBUSxDQUFDLFNBQVMsRUFBRSxDQUFDO1lBQ3hDLElBQUksVUFBVSxDQUFDLE1BQU0sR0FBRyxDQUFDLEVBQUU7Z0JBQ3pCLGdCQUFNLENBQUMsS0FBSyxDQUNWLDRJQUE0SSxVQUFVLEdBQUcsQ0FDMUosQ0FBQzthQUNIO1lBRUQsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLFFBQVEsRUFBRSxRQUFRLEVBQUUsQ0FBQyxDQUFDO1FBQ3JDLENBQUMsQ0FBQyxDQUFDO1FBRUgsT0FBTyxHQUFHLEVBQUU7WUFDVixnQkFBTSxDQUFDLEtBQUssQ0FBQyw4Q0FBOEMsQ0FBQyxDQUFDO1lBQzdELE9BQU8sQ0FBQyxJQUFJLEVBQUUsQ0FBQztRQUNqQixDQUFDLENBQUM7SUFDSixDQUFDLENBQUM7QUFDSixDQUFDO0FBekNELGtEQXlDQztBQUVELFNBQWdCLDJCQUEyQixDQUFDLE9BVTNDOztJQUNDLE1BQU0sT0FBTyxHQUFHLElBQUEsa0JBQVMsRUFBQyxxQkFBYSxDQUFDLENBQUM7SUFDekMsTUFBTSxJQUFJLEdBQUc7UUFDWCxPQUFPO1FBQ1AsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNqRCw4SkFBOEo7UUFFOUosR0FBRyxPQUFPLENBQUMsWUFBWTtRQUN2QixHQUFHLENBQUMsT0FBTyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxlQUFlLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ2pELEdBQUcsQ0FBQyxPQUFPLENBQUMsT0FBTyxDQUFDLENBQUMsQ0FBQyxDQUFDLFdBQVcsQ0FBQyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUM7UUFDekMsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLEdBQUcsQ0FBQyxDQUFDO1lBQzFCLENBQUMsQ0FBQyxDQUFDLGdCQUFnQixPQUFPLENBQUMsV0FBVyxFQUFFLENBQUM7WUFDekMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztLQUNSLENBQUM7SUFFRixNQUFBLE9BQU8sQ0FBQyxJQUFJLDBDQUFFLE9BQU8sQ0FBQyxDQUFDLENBQUMsRUFBRSxFQUFFO1FBQzFCLElBQUksQ0FBQyxJQUFJLENBQUMsUUFBUSxDQUFDLEVBQUUsQ0FBQyxDQUFDO0lBQ3pCLENBQUMsQ0FBQyxDQUFDO0lBRUgsTUFBQSxPQUFPLENBQUMsUUFBUSwwQ0FBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRTtRQUM5QixJQUFJLENBQUMsSUFBSSxDQUFDLGFBQWEsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUM5QixDQUFDLENBQUMsQ0FBQztJQUVILGdCQUFNLENBQUMsS0FBSyxDQUNWLGFBQWEsT0FBTyxDQUFDLG1CQUFtQixJQUFJLElBQUksQ0FBQyxJQUFJLENBQUMsR0FBRyxDQUFDLE9BQ3hELE9BQU8sQ0FBQyxPQUNWLEVBQUUsQ0FDSCxDQUFDO0lBRUYsTUFBTSxNQUFNLEdBQW1CO1FBQzdCLElBQUksRUFBRSxPQUFPLENBQUMsbUJBQW1CO1FBQ2pDLElBQUk7UUFDSixPQUFPLEVBQUU7WUFDUCxHQUFHLEVBQUUsT0FBTyxDQUFDLE9BQU87WUFDcEIsR0FBRyxFQUFFLE9BQU8sQ0FBQyxHQUFnQyxFQUFFLDBEQUEwRDtTQUMxRztLQUNGLENBQUM7SUFFRixPQUFPLENBQUMsSUFBSSxDQUFDLEVBQUUsSUFBSSxFQUFFLE9BQU8sRUFBRSxHQUFHLEVBQUUsTUFBTSxFQUFFLENBQUMsQ0FBQztJQUU3QyxPQUFPLE9BQU8sQ0FBQztBQUNqQixDQUFDO0FBbkRELGtFQW1EQztBQUVELFNBQWdCLDRCQUE0QixDQUFDLE9BUzVDOztJQUNDLE1BQU0sT0FBTyxHQUFHLElBQUEsa0JBQVMsRUFBQyxxQkFBYSxDQUFDLENBQUM7SUFFekMsTUFBTSxJQUFJLEdBQUc7UUFDWCxTQUFTO1FBQ1QsR0FBRyxDQUFDLE9BQU8sQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsZUFBZSxDQUFDLENBQUMsQ0FBQyxDQUFDLEVBQUUsQ0FBQztRQUNqRCw4SkFBOEo7UUFFOUosR0FBRyxPQUFPLENBQUMsWUFBWTtRQUN2QixHQUFHLENBQUMsT0FBTyxDQUFDLE9BQU8sQ0FBQyxDQUFDLENBQUMsQ0FBQyxXQUFXLENBQUMsQ0FBQyxDQUFDLENBQUMsRUFBRSxDQUFDO1FBQ3pDLEdBQUcsQ0FBQyxPQUFPLENBQUMsV0FBVyxHQUFHLENBQUMsQ0FBQztZQUMxQixDQUFDLENBQUMsQ0FBQyxnQkFBZ0IsT0FBTyxDQUFDLFdBQVcsRUFBRSxDQUFDO1lBQ3pDLENBQUMsQ0FBQyxFQUFFLENBQUM7S0FDUixDQUFDO0lBRUYsTUFBQSxPQUFPLENBQUMsSUFBSSwwQ0FBRSxPQUFPLENBQUMsQ0FBQyxDQUFDLEVBQUUsRUFBRTtRQUMxQixJQUFJLENBQUMsSUFBSSxDQUFDLFFBQVEsQ0FBQyxFQUFFLENBQUMsQ0FBQztJQUN6QixDQUFDLENBQUMsQ0FBQztJQUVILE1BQUEsT0FBTyxDQUFDLFFBQVEsMENBQUUsT0FBTyxDQUFDLENBQUMsQ0FBQyxFQUFFLEVBQUU7UUFDOUIsSUFBSSxDQUFDLElBQUksQ0FBQyxhQUFhLENBQUMsRUFBRSxDQUFDLENBQUM7SUFDOUIsQ0FBQyxDQUFDLENBQUM7SUFFSCxnQkFBTSxDQUFDLEtBQUssQ0FDVixhQUFhLE9BQU8sQ0FBQyxtQkFBbUIsSUFBSSxJQUFJLENBQUMsSUFBSSxDQUFDLEdBQUcsQ0FBQyxPQUN4RCxPQUFPLENBQUMsT0FDVixFQUFFLENBQ0gsQ0FBQztJQUVGLE1BQU0sTUFBTSxHQUFtQjtRQUM3QixJQUFJLEVBQUUsT0FBTyxDQUFDLG1CQUFtQjtRQUNqQyxJQUFJO1FBQ0osT0FBTyxFQUFFO1lBQ1AsR0FBRyxFQUFFLE9BQU8sQ0FBQyxPQUFPO1lBQ3BCLEdBQUcsRUFBRSxPQUFPLENBQUMsR0FBZ0MsRUFBRSwwREFBMEQ7U0FDMUc7S0FDRixDQUFDO0lBRUYsT0FBTyxDQUFDLElBQUksQ0FBQyxFQUFFLElBQUksRUFBRSxPQUFPLEVBQUUsR0FBRyxFQUFFLE1BQU0sRUFBRSxDQUFDLENBQUM7SUFFN0MsT0FBTyxPQUFPLENBQUM7QUFDakIsQ0FBQztBQWxERCxvRUFrREMiLCJzb3VyY2VzQ29udGVudCI6WyIvKipcbiAqIENvcHlyaWdodCAoYykgSGFzaGlDb3JwLCBJbmMuXG4gKiBTUERYLUxpY2Vuc2UtSWRlbnRpZmllcjogTVBMLTIuMFxuICovXG5cbmltcG9ydCB7XG4gIGNyZWF0ZU1hY2hpbmUsXG4gIHNlbmQsXG4gIGludGVycHJldCxcbiAgRXZlbnRPYmplY3QsXG4gIGFzc2lnbixcbiAgU2VuZGVyLFxuICBSZWNlaXZlcixcbn0gZnJvbSBcInhzdGF0ZVwiO1xuaW1wb3J0ICogYXMgcHR5IGZyb20gXCJAY2RrdGYvbm9kZS1wdHktcHJlYnVpbHQtbXVsdGlhcmNoXCI7XG5pbXBvcnQgeyBFcnJvcnMsIGxvZ2dlciB9IGZyb20gXCJAY2RrdGYvY29tbW9uc1wiO1xuaW1wb3J0IHsgbWlzc2luZ1ZhcmlhYmxlIH0gZnJvbSBcIi4uL2Vycm9yc1wiO1xuaW1wb3J0IHN0cmlwQW5zaSBmcm9tIFwic3RyaXAtYW5zaVwiO1xuaW1wb3J0IHsgRU9MIH0gZnJvbSBcIm9zXCI7XG5pbXBvcnQgeyBzcGF3blB0eSB9IGZyb20gXCIuL3B0eS1wcm9jZXNzXCI7XG5cbmludGVyZmFjZSBQdHlTcGF3bkNvbmZpZyB7XG4gIGZpbGU6IFBhcmFtZXRlcnM8dHlwZW9mIHB0eS5zcGF3bj5bMF07XG4gIGFyZ3M6IFBhcmFtZXRlcnM8dHlwZW9mIHB0eS5zcGF3bj5bMV07XG4gIG9wdGlvbnM6IFBhcmFtZXRlcnM8dHlwZW9mIHB0eS5zcGF3bj5bMl0gJiB7IGN3ZDogc3RyaW5nIH07XG59XG5cbmludGVyZmFjZSBEZXBsb3lDb250ZXh0IHtcbiAgZXhpdENvZGU/OiBudW1iZXI7XG4gIC8qKlxuICAgKiBUZXJyYWZvcm0gd2lsbCBleGl0IHdpdGggMSBpZiBpdCB3YXMgY2FuY2VsbGVkLCBidXQgd2UgZG9uJ3Qgd2FudCB0byBmYWlsIGluIHRoYXQgY2FzZVxuICAgKi9cbiAgY2FuY2VsbGVkPzogYm9vbGVhbjtcbn1cblxuZXhwb3J0IHR5cGUgRGVwbG95RXZlbnQgPVxuICB8IHsgdHlwZTogXCJTVEFSVFwiOyBwdHk6IFB0eVNwYXduQ29uZmlnIH1cbiAgfCB7IHR5cGU6IFwiU1RPUFwiIH1cbiAgfCB7IHR5cGU6IFwiU0VORF9MSU5FXCI7IGlucHV0OiBzdHJpbmcgfVxuICB8IHsgdHlwZTogXCJPVVRQVVRfUkVDRUlWRURcIjsgb3V0cHV0OiBzdHJpbmcgfVxuICB8IHsgdHlwZTogXCJBUFBST1ZFRF9FWFRFUk5BTExZXCIgfSAvLyBlLmcuIHZpYSBURkMgVUkgb3IgQVBJXG4gIHwgeyB0eXBlOiBcIlJFSkVDVEVEX0VYVEVSTkFMTFlcIiB9XG4gIHwgeyB0eXBlOiBcIk9WRVJSSURERU5fRVhURVJOQUxMWVwiIH1cbiAgfCB7IHR5cGU6IFwiT1ZFUlJJREVfUkVKRUNURURfRVhURVJOQUxMWVwiIH1cbiAgfCB7IHR5cGU6IFwiT1ZFUlJJREVcIiB9XG4gIHwgeyB0eXBlOiBcIlJFSkVDVF9PVkVSUklERVwiIH1cbiAgfCB7IHR5cGU6IFwiUkVRVUVTVF9BUFBST1ZBTFwiIH1cbiAgfCB7IHR5cGU6IFwiVkFSSUFCTEVfTUlTU0lOR1wiOyB2YXJpYWJsZU5hbWU6IHN0cmluZyB9XG4gIHwgeyB0eXBlOiBcIlJFUVVFU1RfU0VOVElORUxfT1ZFUlJJREVcIiB9XG4gIHwgeyB0eXBlOiBcIkFQUFJPVkVcIiB9XG4gIHwgeyB0eXBlOiBcIlJFSkVDVFwiIH1cbiAgfCB7IHR5cGU6IFwiRVhJVEVEXCI7IGV4aXRDb2RlOiBudW1iZXIgfTtcblxuZXhwb3J0IGZ1bmN0aW9uIGlzRGVwbG95RXZlbnQ8RGVwbG95RXZlbnRUeXBlIGV4dGVuZHMgRGVwbG95RXZlbnRbXCJ0eXBlXCJdPihcbiAgZXZlbnQ6IEV2ZW50T2JqZWN0LFxuICB0eXBlOiBEZXBsb3lFdmVudFR5cGUsXG4pOiBldmVudCBpcyBEZXBsb3lFdmVudCAmIHsgdHlwZTogRGVwbG95RXZlbnRUeXBlIH0ge1xuICByZXR1cm4gZXZlbnQudHlwZSA9PT0gdHlwZTtcbn1cblxuZXhwb3J0IHR5cGUgRGVwbG95U3RhdGUgPVxuICB8IHtcbiAgICAgIHZhbHVlOiBcImlkbGVcIjtcbiAgICAgIGNvbnRleHQ6IERlcGxveUNvbnRleHQ7XG4gICAgfVxuICB8IHtcbiAgICAgIHZhbHVlOiBcInJ1bm5pbmdcIjtcbiAgICAgIGNvbnRleHQ6IERlcGxveUNvbnRleHQ7XG4gICAgfVxuICB8IHtcbiAgICAgIHZhbHVlOiB7IHJ1bm5pbmc6IFwicHJvY2Vzc2luZ1wiIH07XG4gICAgICBjb250ZXh0OiBEZXBsb3lDb250ZXh0O1xuICAgIH1cbiAgfCB7XG4gICAgICB2YWx1ZTogeyBydW5uaW5nOiBcImF3YWl0aW5nX2FwcHJvdmFsXCIgfTtcbiAgICAgIGNvbnRleHQ6IERlcGxveUNvbnRleHQ7XG4gICAgfVxuICB8IHtcbiAgICAgIHZhbHVlOiB7IHJ1bm5pbmc6IFwiYXdhaXRpbmdfc2VudGluZWxfb3ZlcnJpZGVcIiB9O1xuICAgICAgY29udGV4dDogRGVwbG95Q29udGV4dDtcbiAgICB9XG4gIHwge1xuICAgICAgdmFsdWU6IFwiZXhpdGVkXCI7XG4gICAgICBjb250ZXh0OiBEZXBsb3lDb250ZXh0ICYgeyBleGl0Q29kZTogbnVtYmVyIH07XG4gICAgfVxuICB8IHtcbiAgICAgIHZhbHVlOiBcInN0b3BwZWRcIjtcbiAgICAgIGNvbnRleHQ6IERlcGxveUNvbnRleHQ7XG4gICAgfTtcblxuZXhwb3J0IGZ1bmN0aW9uIGV4dHJhY3RWYXJpYWJsZU5hbWVGcm9tUHJvbXB0KGxpbmU6IHN0cmluZykge1xuICBjb25zdCBub0NvbG9yTGluZSA9IHN0cmlwQW5zaShsaW5lKTtcbiAgY29uc3QgbGluZXMgPSBub0NvbG9yTGluZS5zcGxpdChcIlxcblwiKTtcbiAgY29uc3QgbGluZVdpdGhWYXIgPSBsaW5lcy5maW5kKChsaW5lKSA9PiBsaW5lLmluY2x1ZGVzKFwidmFyLlwiKSk7XG4gIGlmICghbGluZVdpdGhWYXIpIHtcbiAgICB0aHJvdyBFcnJvcnMuSW50ZXJuYWwoXG4gICAgICBgQ291bGQgbm90IGZpbmQgdmFyaWFibGUgbmFtZSBpbiBwcm9tcHQ6ICR7bGluZX0uIFRoaXMgaXMgbW9zdCBsaWtlbHkgYSBidWcgaW4gY2RrdGYuIFBsZWFzZSByZXBvcnQgaXQgYXQgaHR0cHM6Ly9jZGsudGYvYnVnYCxcbiAgICApO1xuICB9XG4gIHJldHVybiBsaW5lV2l0aFZhci5zcGxpdChcInZhci5cIilbMV0udHJpbSgpO1xufVxuXG5pbnRlcmZhY2UgQnVmZmVyZWRSZWNlaXZlckZ1bmN0aW9uIHtcbiAgKG91dHB1dDogc3RyaW5nKTogdm9pZDtcbiAgLyoqXG4gICAqIHVzZWQgdG8gZ2V0IHRoZSBsYXN0IGJ1ZmZlciB3aGVuIHRoZSBQVFkgZXhpdHMgdG8gbG9nIGEgZGVidWcgbWVzc2FnZSBpZiB0aGVyZSdzIG91dHB1dCBsZWZ0IGluIHRoZXJlXG4gICAqIChtaWdodCBoZWxwIGRlYnVnZ2luZyBvbiBXaW5kb3dzIGlmIEVPTCBmcm9tIE5vZGVqcyBkb2Vzbid0IHdvcmsgaW4gV1NMKVxuICAgKi9cbiAgZ2V0QnVmZmVyOiAoKSA9PiBzdHJpbmc7XG59XG5cbi8vIHVzZWQgdG8gb25seSBzZW5kIGNvbXBsZXRlZCBsaW5lcyAoPSB3aXRoIGEgbmV3bGluZSBhdCB0aGUgZW5kIG9mIHRoZW0pIHRvIG91ciBsb2dpYyAoIzI4MjcpXG5leHBvcnQgZnVuY3Rpb24gYnVmZmVyVW50ZXJtaW5hdGVkTGluZXMoXG4gIGhhbmRsZXI6IChvdXRwdXQ6IHN0cmluZykgPT4gdm9pZCxcbik6IEJ1ZmZlcmVkUmVjZWl2ZXJGdW5jdGlvbiB7XG4gIGxldCBidWZmZXIgPSBcIlwiO1xuICBmdW5jdGlvbiBidWZmZXJlZFJlY2VpdmVyRnVuY3Rpb24ob3V0cHV0OiBzdHJpbmcpIHtcbiAgICBidWZmZXIgKz0gb3V0cHV0O1xuICAgIGNvbnN0IGxpbmVzID0gYnVmZmVyLnNwbGl0KEVPTCk7XG5cbiAgICAvLyBpZiB0aGUgc3RyaW5nIGVuZHMgd2l0aCBcXG4gdGhpcyB3aWxsIGJlIGFuIGVtcHR5IHN0cmluZ1xuICAgIC8vIGVsc2UgaXQgd2lsbCBjb250YWluIGFuIFwidW5maW5pc2hlZFwiIGxpbmVcbiAgICAvLyB0aGUgZmFsbGJhY2sgdG8gYW4gZW1wdHkgc3RyaW5nIGlzIHRvIG1ha2UgVFMgaGFwcHkgYW5kIHNob3VsZCBuZXZlciBoYXBwZW5cbiAgICBidWZmZXIgPSBsaW5lcy5wb3AoKSB8fCBcIlwiO1xuXG4gICAgaWYgKGxpbmVzLmxlbmd0aCA+IDApIGhhbmRsZXIobGluZXMuam9pbihFT0wpICsgXCJcXG5cIik7XG4gIH1cbiAgYnVmZmVyZWRSZWNlaXZlckZ1bmN0aW9uLmdldEJ1ZmZlciA9ICgpID0+IGJ1ZmZlcjtcblxuICByZXR1cm4gYnVmZmVyZWRSZWNlaXZlckZ1bmN0aW9uO1xufVxuXG5leHBvcnQgZnVuY3Rpb24gaGFuZGxlTGluZVJlY2VpdmVkKHNlbmQ6IChldmVudDogRGVwbG95RXZlbnQpID0+IHZvaWQpIHtcbiAgcmV0dXJuIChvdXRwdXQ6IHN0cmluZykgPT4ge1xuICAgIGxldCBoaWRlT3V0cHV0ID0gZmFsc2U7XG4gICAgY29uc3Qgbm9Db2xvckxpbmUgPSBzdHJpcEFuc2kob3V0cHV0KTtcblxuICAgIC8vIHBvc3NpYmxlIGV2ZW50cyBiYXNlZCBvbiBsaW5lXG4gICAgaWYgKG5vQ29sb3JMaW5lLmluY2x1ZGVzKFwiYXBwcm92ZWQgdXNpbmcgdGhlIFVJIG9yIEFQSVwiKSkge1xuICAgICAgc2VuZCh7IHR5cGU6IFwiQVBQUk9WRURfRVhURVJOQUxMWVwiIH0pO1xuICAgIH0gZWxzZSBpZiAobm9Db2xvckxpbmUuaW5jbHVkZXMoXCJkaXNjYXJkZWQgdXNpbmcgdGhlIFVJIG9yIEFQSVwiKSkge1xuICAgICAgc2VuZCh7IHR5cGU6IFwiUkVKRUNURURfRVhURVJOQUxMWVwiIH0pO1xuICAgIH0gZWxzZSBpZiAoXG4gICAgICBub0NvbG9yTGluZS5pbmNsdWRlcyhcIkRvIHlvdSB3YW50IHRvIHBlcmZvcm0gdGhlc2UgYWN0aW9uc1wiKSB8fFxuICAgICAgbm9Db2xvckxpbmUuaW5jbHVkZXMoXCJEbyB5b3UgcmVhbGx5IHdhbnQgdG8gZGVzdHJveSBhbGwgcmVzb3VyY2VzP1wiKSB8fFxuICAgICAgbm9Db2xvckxpbmUuaW5jbHVkZXMoXG4gICAgICAgIFwiRG8geW91IHJlYWxseSB3YW50IHRvIGRlc3Ryb3kgYWxsIHJlc291cmNlcyBpbiB3b3Jrc3BhY2VcIixcbiAgICAgIClcbiAgICApIHtcbiAgICAgIGhpZGVPdXRwdXQgPSB0cnVlO1xuICAgICAgc2VuZCh7IHR5cGU6IFwiT1VUUFVUX1JFQ0VJVkVEXCIsIG91dHB1dCB9KTtcbiAgICAgIHNlbmQoeyB0eXBlOiBcIlJFUVVFU1RfQVBQUk9WQUxcIiB9KTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgbm9Db2xvckxpbmUuaW5jbHVkZXMoXCJ2YXIuXCIpICYmXG4gICAgICBub0NvbG9yTGluZS5pbmNsdWRlcyhcIkVudGVyIGEgdmFsdWU6XCIpXG4gICAgKSB7XG4gICAgICBoaWRlT3V0cHV0ID0gdHJ1ZTtcblxuICAgICAgY29uc3QgdmFyaWFibGVOYW1lID0gZXh0cmFjdFZhcmlhYmxlTmFtZUZyb21Qcm9tcHQob3V0cHV0KTtcbiAgICAgIHNlbmQoe1xuICAgICAgICB0eXBlOiBcIk9VVFBVVF9SRUNFSVZFRFwiLFxuICAgICAgICBvdXRwdXQ6IG1pc3NpbmdWYXJpYWJsZSh2YXJpYWJsZU5hbWUpLFxuICAgICAgfSk7XG4gICAgICBzZW5kKHsgdHlwZTogXCJWQVJJQUJMRV9NSVNTSU5HXCIsIHZhcmlhYmxlTmFtZSB9KTtcbiAgICB9IGVsc2UgaWYgKFxuICAgICAgbm9Db2xvckxpbmUuaW5jbHVkZXMoXG4gICAgICAgIFwiRG8geW91IHdhbnQgdG8gb3ZlcnJpZGUgdGhlIHNvZnQgZmFpbGVkIHBvbGljeSBjaGVjaz9cIixcbiAgICAgIClcbiAgICApIHtcbiAgICAgIGhpZGVPdXRwdXQgPSB0cnVlO1xuICAgICAgc2VuZCh7IHR5cGU6IFwiT1VUUFVUX1JFQ0VJVkVEXCIsIG91dHB1dCB9KTtcblxuICAgICAgc2VuZCh7IHR5cGU6IFwiUkVRVUVTVF9TRU5USU5FTF9PVkVSUklERVwiIH0pO1xuICAgIH0gZWxzZSBpZiAobm9Db2xvckxpbmUuaW5jbHVkZXMoXCJvdmVycmlkZGVuIHVzaW5nIHRoZSBVSSBvciBBUElcIikpIHtcbiAgICAgIHNlbmQoeyB0eXBlOiBcIk9WRVJSSURERU5fRVhURVJOQUxMWVwiIH0pO1xuICAgIH1cblxuICAgIGlmICghaGlkZU91dHB1dCkge1xuICAgICAgc2VuZCh7XG4gICAgICAgIHR5cGU6IFwiT1VUUFVUX1JFQ0VJVkVEXCIsXG4gICAgICAgIG91dHB1dCxcbiAgICAgIH0pO1xuICAgIH1cbiAgfTtcbn1cblxuZXhwb3J0IGNvbnN0IGRlcGxveU1hY2hpbmUgPSBjcmVhdGVNYWNoaW5lPFxuICBEZXBsb3lDb250ZXh0LFxuICBEZXBsb3lFdmVudCxcbiAgRGVwbG95U3RhdGVcbj4oXG4gIHtcbiAgICBwcmVkaWN0YWJsZUFjdGlvbkFyZ3VtZW50czogdHJ1ZSxcbiAgICBjb250ZXh0OiB7fSxcbiAgICBpbml0aWFsOiBcImlkbGVcIixcbiAgICBpZDogXCJyb290XCIsXG4gICAgc3RhdGVzOiB7XG4gICAgICBpZGxlOiB7XG4gICAgICAgIG9uOiB7XG4gICAgICAgICAgU1RBUlQ6IHsgdGFyZ2V0OiBcInJ1bm5pbmdcIiB9LFxuICAgICAgICB9LFxuICAgICAgfSxcbiAgICAgIHJ1bm5pbmc6IHtcbiAgICAgICAgaW52b2tlOiB7XG4gICAgICAgICAgaWQ6IFwicHR5XCIsXG4gICAgICAgICAgc3JjOiBcInJ1blRlcnJhZm9ybUluUHR5XCIsXG4gICAgICAgIH0sXG4gICAgICAgIG9uOiB7XG4gICAgICAgICAgRVhJVEVEOiBcImV4aXRlZFwiLFxuICAgICAgICAgIFNUT1A6IFwic3RvcHBlZFwiLFxuICAgICAgICB9LFxuICAgICAgICBpbml0aWFsOiBcInByb2Nlc3NpbmdcIixcbiAgICAgICAgc3RhdGVzOiB7XG4gICAgICAgICAgLy8gVE9ETzogd2hhdCBlbHNlIG1pZ2h0IFRGIENMSSBiZSBhc2tpbmc/IENhbiB3ZSBkZXRlY3QgYW55IHF1ZXN0aW9uIGZyb20gdGhlIFRGIENMSSB0byBzaG93IGEgZ29vZCBlcnJvcj9cbiAgICAgICAgICBwcm9jZXNzaW5nOiB7XG4gICAgICAgICAgICBvbjoge1xuICAgICAgICAgICAgICBSRVFVRVNUX0FQUFJPVkFMOiBcImF3YWl0aW5nX2FwcHJvdmFsXCIsXG4gICAgICAgICAgICAgIFJFUVVFU1RfU0VOVElORUxfT1ZFUlJJREU6IFwiYXdhaXRpbmdfc2VudGluZWxfb3ZlcnJpZGVcIixcbiAgICAgICAgICAgICAgVkFSSUFCTEVfTUlTU0lORzoge1xuICAgICAgICAgICAgICAgIGFjdGlvbnM6IHNlbmQoeyB0eXBlOiBcIkVYSVRFRFwiLCBleGl0Q29kZTogMSB9KSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgICBhd2FpdGluZ19hcHByb3ZhbDoge1xuICAgICAgICAgICAgb246IHtcbiAgICAgICAgICAgICAgQVBQUk9WRURfRVhURVJOQUxMWTogXCJwcm9jZXNzaW5nXCIsXG4gICAgICAgICAgICAgIFJFSkVDVEVEX0VYVEVSTkFMTFk6IHtcbiAgICAgICAgICAgICAgICB0YXJnZXQ6IFwiI3Jvb3QuZXhpdGVkXCIsXG4gICAgICAgICAgICAgICAgYWN0aW9uczogYXNzaWduPFxuICAgICAgICAgICAgICAgICAgRGVwbG95Q29udGV4dCxcbiAgICAgICAgICAgICAgICAgIERlcGxveUV2ZW50ICYgeyB0eXBlOiBcIlJFSkVDVEVEX0VYVEVSTkFMTFlcIiB9XG4gICAgICAgICAgICAgICAgPih7IGNhbmNlbGxlZDogdHJ1ZSB9KSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgQVBQUk9WRToge1xuICAgICAgICAgICAgICAgIHRhcmdldDogXCJwcm9jZXNzaW5nXCIsXG4gICAgICAgICAgICAgICAgYWN0aW9uczogc2VuZChcbiAgICAgICAgICAgICAgICAgIHsgdHlwZTogXCJTRU5EX0xJTkVcIiwgaW5wdXQ6IFwieWVzXCIgfSxcbiAgICAgICAgICAgICAgICAgIHsgdG86IFwicHR5XCIgfSxcbiAgICAgICAgICAgICAgICApLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgICBSRUpFQ1Q6IHtcbiAgICAgICAgICAgICAgICB0YXJnZXQ6IFwicHJvY2Vzc2luZ1wiLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IFtcbiAgICAgICAgICAgICAgICAgIHNlbmQoeyB0eXBlOiBcIlNFTkRfTElORVwiLCBpbnB1dDogXCJub1wiIH0sIHsgdG86IFwicHR5XCIgfSksXG4gICAgICAgICAgICAgICAgICBhc3NpZ248RGVwbG95Q29udGV4dCwgRGVwbG95RXZlbnQgJiB7IHR5cGU6IFwiUkVKRUNUXCIgfT4oe1xuICAgICAgICAgICAgICAgICAgICBjYW5jZWxsZWQ6IHRydWUsXG4gICAgICAgICAgICAgICAgICB9KSxcbiAgICAgICAgICAgICAgICBdLFxuICAgICAgICAgICAgICB9LFxuICAgICAgICAgICAgfSxcbiAgICAgICAgICB9LFxuICAgICAgICAgIGF3YWl0aW5nX3NlbnRpbmVsX292ZXJyaWRlOiB7XG4gICAgICAgICAgICBvbjoge1xuICAgICAgICAgICAgICBPVkVSUklEREVOX0VYVEVSTkFMTFk6IFwicHJvY2Vzc2luZ1wiLFxuICAgICAgICAgICAgICBPVkVSUklERV9SRUpFQ1RFRF9FWFRFUk5BTExZOiB7XG4gICAgICAgICAgICAgICAgdGFyZ2V0OiBcIiNyb290LmV4aXRlZFwiLFxuICAgICAgICAgICAgICAgIGFjdGlvbnM6IGFzc2lnbjxcbiAgICAgICAgICAgICAgICAgIERlcGxveUNvbnRleHQsXG4gICAgICAgICAgICAgICAgICBEZXBsb3lFdmVudCAmIHsgdHlwZTogXCJPVkVSUklERV9SRUpFQ1RFRF9FWFRFUk5BTExZXCIgfVxuICAgICAgICAgICAgICAgID4oeyBjYW5jZWxsZWQ6IHRydWUgfSksXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIC8vIFRoaXMgaXMgYSBiaXQgb2YgYSBoYWNrLCBiZWNhdXNlIHRoZSBleHRlcm5hbCBkaXNjYXJkIG1lc3NhZ2VcbiAgICAgICAgICAgICAgLy8gcG9zdGVkIGJ5IFRlcnJhZm9ybSBVSSBpcyB0aGUgc2FtZSBhcyBkdXJpbmcgYXBwbHkuIFNvLCB3ZSBjYXB0dXJlIHRoYXRcbiAgICAgICAgICAgICAgLy8gYW5kIGVtaXQgb3VyIG93biBldmVudCB0byBtYWtlIGl0IG1vcmUgc3BlY2lmaWMuXG4gICAgICAgICAgICAgIFJFSkVDVEVEX0VYVEVSTkFMTFk6IHtcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBzZW5kKHsgdHlwZTogXCJPVkVSUklERV9SRUpFQ1RFRF9FWFRFUk5BTExZXCIgfSksXG4gICAgICAgICAgICAgIH0sXG4gICAgICAgICAgICAgIE9WRVJSSURFOiB7XG4gICAgICAgICAgICAgICAgdGFyZ2V0OiBcInByb2Nlc3NpbmdcIixcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBzZW5kKFxuICAgICAgICAgICAgICAgICAgeyB0eXBlOiBcIlNFTkRfTElORVwiLCBpbnB1dDogXCJvdmVycmlkZVwiIH0sXG4gICAgICAgICAgICAgICAgICB7IHRvOiBcInB0eVwiIH0sXG4gICAgICAgICAgICAgICAgKSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgICAgUkVKRUNUX09WRVJSSURFOiB7XG4gICAgICAgICAgICAgICAgdGFyZ2V0OiBcInByb2Nlc3NpbmdcIixcbiAgICAgICAgICAgICAgICBhY3Rpb25zOiBbXG4gICAgICAgICAgICAgICAgICBzZW5kKHsgdHlwZTogXCJTRU5EX0xJTkVcIiwgaW5wdXQ6IFwibm9cIiB9LCB7IHRvOiBcInB0eVwiIH0pLFxuICAgICAgICAgICAgICAgICAgYXNzaWduPFxuICAgICAgICAgICAgICAgICAgICBEZXBsb3lDb250ZXh0LFxuICAgICAgICAgICAgICAgICAgICBEZXBsb3lFdmVudCAmIHsgdHlwZTogXCJSRUpFQ1RfT1ZFUlJJREVcIiB9XG4gICAgICAgICAgICAgICAgICA+KHtcbiAgICAgICAgICAgICAgICAgICAgY2FuY2VsbGVkOiB0cnVlLFxuICAgICAgICAgICAgICAgICAgfSksXG4gICAgICAgICAgICAgICAgXSxcbiAgICAgICAgICAgICAgfSxcbiAgICAgICAgICAgIH0sXG4gICAgICAgICAgfSxcbiAgICAgICAgfSxcbiAgICAgIH0sXG4gICAgICBleGl0ZWQ6IHsgdHlwZTogXCJmaW5hbFwiIH0sXG4gICAgICBzdG9wcGVkOiB7IHR5cGU6IFwiZmluYWxcIiB9LFxuICAgIH0sXG4gIH0sXG4gIHtcbiAgICBzZXJ2aWNlczoge1xuICAgICAgcnVuVGVycmFmb3JtSW5QdHk6IChjb250ZXh0LCBldmVudCkgPT5cbiAgICAgICAgdGVycmFmb3JtUHR5U2VydmljZShjb250ZXh0LCBldmVudCwgc3Bhd25QdHkpLFxuICAgIH0sXG4gIH0sXG4pO1xuXG5leHBvcnQgZnVuY3Rpb24gdGVycmFmb3JtUHR5U2VydmljZShcbiAgX2NvbnRleHQ6IERlcGxveUNvbnRleHQsXG4gIGV2ZW50OiBEZXBsb3lFdmVudCxcbiAgc3Bhd24gPSBzcGF3blB0eSxcbik6IChzZW5kOiBTZW5kZXI8RGVwbG95RXZlbnQ+LCBvblJlY2VpdmU6IFJlY2VpdmVyPERlcGxveUV2ZW50PikgPT4gdm9pZCB7XG4gIHJldHVybiAoc2VuZDogU2VuZGVyPERlcGxveUV2ZW50Piwgb25SZWNlaXZlOiBSZWNlaXZlcjxEZXBsb3lFdmVudD4pID0+IHtcbiAgICBpZiAoZXZlbnQudHlwZSAhPT0gXCJTVEFSVFwiKSB7XG4gICAgICB0aHJvdyBFcnJvcnMuSW50ZXJuYWwoXG4gICAgICAgIGBUZXJyYWZvcm0gQ0xJIGludm9jYXRpb24gc3RhdGUgbWFjaGluZTogVW5leHBlY3RlZCBldmVudCBjYXVzZWQgdHJhbnNpdGlvbiB0byB0aGUgcnVubmluZyBzdGF0ZTogJHtldmVudC50eXBlfWAsXG4gICAgICApO1xuICAgIH1cblxuICAgIC8vIENvbW11bmljYXRpb24gZnJvbSB0aGUgcHR5IHRvIHRoZSBjYWxsZXJcbiAgICBjb25zdCByZWNlaXZlciA9IGJ1ZmZlclVudGVybWluYXRlZExpbmVzKGhhbmRsZUxpbmVSZWNlaXZlZChzZW5kKSk7XG4gICAgY29uc3QgeyBleGl0Q29kZSwgYWN0aW9ucyB9ID0gc3Bhd24oZXZlbnQucHR5LCAoZGF0YSkgPT4ge1xuICAgICAgcmVjZWl2ZXIoZGF0YSk7XG4gICAgfSk7XG5cbiAgICAvLyBDb21tdW5pY2F0aW9uIGZyb20gdGhlIGNhbGxlciB0byB0aGUgcHR5XG4gICAgb25SZWNlaXZlKChldmVudDogRGVwbG95RXZlbnQpID0+IHtcbiAgICAgIGlmIChldmVudC50eXBlID09PSBcIlNFTkRfTElORVwiKSB7XG4gICAgICAgIGFjdGlvbnMud3JpdGVMaW5lKGV2ZW50LmlucHV0KTtcbiAgICAgIH1cbiAgICB9KTtcblxuICAgIGV4aXRDb2RlLnRoZW4oKGV4aXRDb2RlKSA9PiB7XG4gICAgICBjb25zdCBsYXN0QnVmZmVyID0gcmVjZWl2ZXIuZ2V0QnVmZmVyKCk7XG4gICAgICBpZiAobGFzdEJ1ZmZlci5sZW5ndGggPiAwKSB7XG4gICAgICAgIGxvZ2dlci5kZWJ1ZyhcbiAgICAgICAgICBgVGVycmFmb3JtIENMSSBleGl0ZWQgYnV0IHRoZSBsYXN0IG91dHB1dHRlZCBsaW5lIHdhcyBub3QgdGVybWluYXRlZCB3aXRoIGEgbmV3bGluZSBhbmQgaGVuY2UgaXMgc3RpbGwgaW4gdGhlIGJ1ZmZlciBhbmQgd2Fzbid0IHByaW50ZWQ6IFwiJHtsYXN0QnVmZmVyfVwiYCxcbiAgICAgICAgKTtcbiAgICAgIH1cblxuICAgICAgc2VuZCh7IHR5cGU6IFwiRVhJVEVEXCIsIGV4aXRDb2RlIH0pO1xuICAgIH0pO1xuXG4gICAgcmV0dXJuICgpID0+IHtcbiAgICAgIGxvZ2dlci50cmFjZShcIlRlcnJhZm9ybSBDTEkgc3RhdGUgbWFjaGluZTogY2xlYW5pbmcgdXAgcHR5XCIpO1xuICAgICAgYWN0aW9ucy5zdG9wKCk7XG4gICAgfTtcbiAgfTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUFuZFN0YXJ0RGVwbG95U2VydmljZShvcHRpb25zOiB7XG4gIHJlZnJlc2hPbmx5PzogYm9vbGVhbjtcbiAgcGFyYWxsZWxpc206IG51bWJlcjtcbiAgZXh0cmFPcHRpb25zOiBzdHJpbmdbXTtcbiAgdGVycmFmb3JtQmluYXJ5TmFtZTogc3RyaW5nO1xuICBhdXRvQXBwcm92ZT86IGJvb2xlYW47XG4gIG5vQ29sb3I/OiBib29sZWFuO1xuICB3b3JrZGlyOiBzdHJpbmc7XG4gIHZhcnM/OiBzdHJpbmdbXTtcbiAgdmFyRmlsZXM/OiBzdHJpbmdbXTtcbn0pIHtcbiAgY29uc3Qgc2VydmljZSA9IGludGVycHJldChkZXBsb3lNYWNoaW5lKTtcbiAgY29uc3QgYXJncyA9IFtcbiAgICBcImFwcGx5XCIsXG4gICAgLi4uKG9wdGlvbnMuYXV0b0FwcHJvdmUgPyBbXCItYXV0by1hcHByb3ZlXCJdIDogW10pLFxuICAgIC8vIFwiLWlucHV0PWZhbHNlXCIsIHdlIGNhbid0IHVzZSB0aGlzIGFueW1vcmUgYnV0IFRPRE86IHdlIG5lZWQgdG8gZGV0ZWN0IFRGIENMSSBhc2tpbmcgZm9yIG1pc3NpbmcgaW5wdXRzIGFuZCBlaXRoZXIgYWxsb3cgcGFzc2luZyB0aGVtIG9yIHN0b3AgdGhlcmUgYW5kIGZhaWxcblxuICAgIC4uLm9wdGlvbnMuZXh0cmFPcHRpb25zLFxuICAgIC4uLihvcHRpb25zLnJlZnJlc2hPbmx5ID8gW1wiLXJlZnJlc2gtb25seVwiXSA6IFtdKSxcbiAgICAuLi4ob3B0aW9ucy5ub0NvbG9yID8gW1wiLW5vLWNvbG9yXCJdIDogW10pLFxuICAgIC4uLihvcHRpb25zLnBhcmFsbGVsaXNtID4gLTFcbiAgICAgID8gW2AtcGFyYWxsZWxpc209JHtvcHRpb25zLnBhcmFsbGVsaXNtfWBdXG4gICAgICA6IFtdKSxcbiAgXTtcblxuICBvcHRpb25zLnZhcnM/LmZvckVhY2goKHYpID0+IHtcbiAgICBhcmdzLnB1c2goYC12YXI9JHt2fWApO1xuICB9KTtcblxuICBvcHRpb25zLnZhckZpbGVzPy5mb3JFYWNoKCh2KSA9PiB7XG4gICAgYXJncy5wdXNoKGAtdmFyLWZpbGU9JHt2fWApO1xuICB9KTtcblxuICBsb2dnZXIuZGVidWcoXG4gICAgYEV4ZWN1dGluZyAke29wdGlvbnMudGVycmFmb3JtQmluYXJ5TmFtZX0gJHthcmdzLmpvaW4oXCIgXCIpfSBpbiAke1xuICAgICAgb3B0aW9ucy53b3JrZGlyXG4gICAgfWAsXG4gICk7XG5cbiAgY29uc3QgY29uZmlnOiBQdHlTcGF3bkNvbmZpZyA9IHtcbiAgICBmaWxlOiBvcHRpb25zLnRlcnJhZm9ybUJpbmFyeU5hbWUsXG4gICAgYXJncyxcbiAgICBvcHRpb25zOiB7XG4gICAgICBjd2Q6IG9wdGlvbnMud29ya2RpcixcbiAgICAgIGVudjogcHJvY2Vzcy5lbnYgYXMgeyBba2V5OiBzdHJpbmddOiBzdHJpbmcgfSwgLy8gVE9ETzogbWFrZSB0aGlzIGV4cGxpY2l0IGFuZCBtb3ZlIHRvIGNhbGxlciBvciB3aGF0ZXZlclxuICAgIH0sXG4gIH07XG5cbiAgc2VydmljZS5zZW5kKHsgdHlwZTogXCJTVEFSVFwiLCBwdHk6IGNvbmZpZyB9KTtcblxuICByZXR1cm4gc2VydmljZTtcbn1cblxuZXhwb3J0IGZ1bmN0aW9uIGNyZWF0ZUFuZFN0YXJ0RGVzdHJveVNlcnZpY2Uob3B0aW9uczoge1xuICBwYXJhbGxlbGlzbTogbnVtYmVyO1xuICBleHRyYU9wdGlvbnM6IHN0cmluZ1tdO1xuICB0ZXJyYWZvcm1CaW5hcnlOYW1lOiBzdHJpbmc7XG4gIGF1dG9BcHByb3ZlPzogYm9vbGVhbjtcbiAgbm9Db2xvcj86IGJvb2xlYW47XG4gIHdvcmtkaXI6IHN0cmluZztcbiAgdmFycz86IHN0cmluZ1tdO1xuICB2YXJGaWxlcz86IHN0cmluZ1tdO1xufSkge1xuICBjb25zdCBzZXJ2aWNlID0gaW50ZXJwcmV0KGRlcGxveU1hY2hpbmUpO1xuXG4gIGNvbnN0IGFyZ3MgPSBbXG4gICAgXCJkZXN0cm95XCIsXG4gICAgLi4uKG9wdGlvbnMuYXV0b0FwcHJvdmUgPyBbXCItYXV0by1hcHByb3ZlXCJdIDogW10pLFxuICAgIC8vIFwiLWlucHV0PWZhbHNlXCIsIHdlIGNhbid0IHVzZSB0aGlzIGFueW1vcmUgYnV0IFRPRE86IHdlIG5lZWQgdG8gZGV0ZWN0IFRGIENMSSBhc2tpbmcgZm9yIG1pc3NpbmcgaW5wdXRzIGFuZCBlaXRoZXIgYWxsb3cgcGFzc2luZyB0aGVtIG9yIHN0b3AgdGhlcmUgYW5kIGZhaWxcblxuICAgIC4uLm9wdGlvbnMuZXh0cmFPcHRpb25zLFxuICAgIC4uLihvcHRpb25zLm5vQ29sb3IgPyBbXCItbm8tY29sb3JcIl0gOiBbXSksXG4gICAgLi4uKG9wdGlvbnMucGFyYWxsZWxpc20gPiAtMVxuICAgICAgPyBbYC1wYXJhbGxlbGlzbT0ke29wdGlvbnMucGFyYWxsZWxpc219YF1cbiAgICAgIDogW10pLFxuICBdO1xuXG4gIG9wdGlvbnMudmFycz8uZm9yRWFjaCgodikgPT4ge1xuICAgIGFyZ3MucHVzaChgLXZhcj0ke3Z9YCk7XG4gIH0pO1xuXG4gIG9wdGlvbnMudmFyRmlsZXM/LmZvckVhY2goKHYpID0+IHtcbiAgICBhcmdzLnB1c2goYC12YXItZmlsZT0ke3Z9YCk7XG4gIH0pO1xuXG4gIGxvZ2dlci5kZWJ1ZyhcbiAgICBgRXhlY3V0aW5nICR7b3B0aW9ucy50ZXJyYWZvcm1CaW5hcnlOYW1lfSAke2FyZ3Muam9pbihcIiBcIil9IGluICR7XG4gICAgICBvcHRpb25zLndvcmtkaXJcbiAgICB9YCxcbiAgKTtcblxuICBjb25zdCBjb25maWc6IFB0eVNwYXduQ29uZmlnID0ge1xuICAgIGZpbGU6IG9wdGlvbnMudGVycmFmb3JtQmluYXJ5TmFtZSxcbiAgICBhcmdzLFxuICAgIG9wdGlvbnM6IHtcbiAgICAgIGN3ZDogb3B0aW9ucy53b3JrZGlyLFxuICAgICAgZW52OiBwcm9jZXNzLmVudiBhcyB7IFtrZXk6IHN0cmluZ106IHN0cmluZyB9LCAvLyBUT0RPOiBtYWtlIHRoaXMgZXhwbGljaXQgYW5kIG1vdmUgdG8gY2FsbGVyIG9yIHdoYXRldmVyXG4gICAgfSxcbiAgfTtcblxuICBzZXJ2aWNlLnNlbmQoeyB0eXBlOiBcIlNUQVJUXCIsIHB0eTogY29uZmlnIH0pO1xuXG4gIHJldHVybiBzZXJ2aWNlO1xufVxuIl19