"use strict";
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.checkIfAllDependenciesAreIncluded = exports.checkIfAllDependantsAreIncluded = exports.getStackWithNoUnmetDependants = exports.findAllNestedDependantStacks = exports.getStackWithNoUnmetDependencies = exports.getMultipleStacks = exports.getSingleStack = void 0;
const commons_1 = require("@cdktf/commons");
const minimatch_1 = __importDefault(require("minimatch"));
function getSingleStack(stacks, stackName, targetAction) {
    if (!stacks) {
        throw commons_1.Errors.Internal("Trying to access a stack before it has been synthesized");
    }
    if (stackName) {
        const stack = stacks.find((s) => s.name === stackName);
        if (!stack) {
            throw commons_1.Errors.Usage("Could not find stack: " + stackName);
        }
        return stack;
    }
    if (stacks.length === 1) {
        return stacks[0];
    }
    throw commons_1.Errors.Usage(`Found more than one stack, please specify a target stack. Run cdktf ${targetAction || "<verb>"} <stack> with one of these stacks: ${stacks
        .map((s) => s.name)
        .join(", ")} `);
}
exports.getSingleStack = getSingleStack;
function getMultipleStacks(stacks, patterns, targetAction) {
    if (!patterns || !patterns.length) {
        if (stacks.length === 1) {
            return [stacks[0]];
        }
        throw commons_1.Errors.Usage(`Found more than one stack, please specify a target stack. Run cdktf ${targetAction || "<verb>"} <stack> with one of these stacks: ${stacks
            .map((s) => s.name)
            .join(", ")} `);
    }
    return patterns.flatMap((pattern) => {
        const matchingStacks = stacks.filter((stack) => (0, minimatch_1.default)(stack.name, pattern));
        if (matchingStacks.length === 0) {
            throw commons_1.Errors.Usage(`Could not find stack for pattern '${pattern}'`);
        }
        return matchingStacks;
    });
}
exports.getMultipleStacks = getMultipleStacks;
// Returns the first stack that has no unmet dependencies
// An unmet dependency is a dependency that has not been deployed yet
// If there is no unfinished stack, returns undefined
// If there is no stack ready to be worked on, it returns a promise that will resolve as soon as there is a follow-up stack available
async function getStackWithNoUnmetDependencies(stackExecutors) {
    commons_1.logger.debug("Checking for stacks with unmet dependencies");
    commons_1.logger.debug("stack executors:", stackExecutors);
    const pendingStacks = stackExecutors.filter((executor) => executor.isPending);
    commons_1.logger.debug("pending stacks:", stackExecutors);
    if (pendingStacks.length === 0) {
        return undefined;
    }
    const currentlyReadyStack = pendingStacks.find((executor) => executor.stack.dependencies.every((dependency) => {
        var _a;
        return ((_a = stackExecutors.find((executor) => executor.stack.name === dependency)) === null || _a === void 0 ? void 0 : _a.currentState) === "done";
    }));
    if (currentlyReadyStack) {
        commons_1.logger.debug("Found a stack with no unmet dependencies");
        return currentlyReadyStack;
    }
    const stackExecutionPromises = stackExecutors
        .filter((ex) => ex.currentWorkPromise)
        .map((ex) => ex.currentWorkPromise);
    commons_1.logger.debug(`${stackExecutionPromises.length} stacks are currently busy, waiting for one to finish`);
    if (!stackExecutionPromises.length) {
        return undefined;
    }
    await (0, commons_1.ensureAllSettledBeforeThrowing)(Promise.race(stackExecutionPromises), stackExecutionPromises);
    return await getStackWithNoUnmetDependencies(stackExecutors);
}
exports.getStackWithNoUnmetDependencies = getStackWithNoUnmetDependencies;
function findAllDependantStacks(stackExecutors, stackName) {
    return stackExecutors.filter((innerExecutor) => innerExecutor.stack.dependencies.includes(stackName));
}
function findAllNestedDependantStacks(stackExecutors, stackName, knownDependantStackNames = new Set()) {
    const dependantStacks = findAllDependantStacks(stackExecutors, stackName);
    dependantStacks.forEach((stack) => {
        if (knownDependantStackNames.has(stack.stack.name)) {
            return;
        }
        knownDependantStackNames.add(stack.stack.name);
        findAllNestedDependantStacks(stackExecutors, stack.stack.name, knownDependantStackNames);
    });
    return stackExecutors.filter((executor) => knownDependantStackNames.has(executor.stack.name));
}
exports.findAllNestedDependantStacks = findAllNestedDependantStacks;
// Returns the first stack that has no dependents that need to be destroyed first
async function getStackWithNoUnmetDependants(stackExecutors) {
    commons_1.logger.debug("Checking for stacks with unmet dependants");
    commons_1.logger.debug("stack executors:", stackExecutors);
    const pendingStacks = stackExecutors.filter((executor) => executor.isPending);
    commons_1.logger.debug("pending stacks:", stackExecutors);
    if (pendingStacks.length === 0) {
        return undefined;
    }
    const currentlyReadyStack = pendingStacks.find((executor) => {
        const dependantStacks = findAllDependantStacks(stackExecutors, executor.stack.name);
        return dependantStacks.every((stack) => stack.currentState === "done");
    });
    if (currentlyReadyStack) {
        commons_1.logger.debug("Found a stack with no unmet dependants");
        return currentlyReadyStack;
    }
    const stackExecutionPromises = stackExecutors
        .filter((ex) => ex.currentWorkPromise)
        .map((ex) => ex.currentWorkPromise);
    commons_1.logger.debug(`${stackExecutionPromises.length} stacks are currently busy, waiting for one to finish`);
    if (!stackExecutionPromises.length) {
        return undefined;
    }
    await Promise.race(stackExecutionPromises);
    return await getStackWithNoUnmetDependants(stackExecutors);
}
exports.getStackWithNoUnmetDependants = getStackWithNoUnmetDependants;
// Throws an error if there is a dependant stack that is not included
function checkIfAllDependantsAreIncluded(stacksToRun, allStacks) {
    const allDependants = new Set();
    stacksToRun
        .map((stack) => allStacks.filter((s) => s.dependencies.includes(stack.name)))
        .flat()
        .forEach((dependant) => allDependants.add(dependant.name));
    const stackNames = stacksToRun.map((stack) => stack.name);
    const missingDependants = [...allDependants].filter((dependant) => !stackNames.includes(dependant));
    if (missingDependants.length > 0) {
        throw commons_1.Errors.Usage(`The following dependant stacks are not included in the stacks to run: ${missingDependants.join(", ")}. Either add them or add the --ignore-missing-stack-dependencies flag.`);
    }
}
exports.checkIfAllDependantsAreIncluded = checkIfAllDependantsAreIncluded;
/**
  Throws an error if there is a dependency that is not included
  Cycles are detected on dependency creation at synthesis time
  Running this prevents us from being in a situation where we
  have to wait for a stack to be deployed that is not included to be run
*/
function checkIfAllDependenciesAreIncluded(stacksToRun) {
    const allDependencies = new Set();
    stacksToRun
        .map((stack) => stack.dependencies)
        .flat()
        .forEach((dependency) => allDependencies.add(dependency));
    const stackNames = stacksToRun.map((stack) => stack.name);
    const missingDependencies = [...allDependencies].filter((dependency) => !stackNames.includes(dependency));
    if (missingDependencies.length > 0) {
        throw commons_1.Errors.Usage(`The following dependencies are not included in the stacks to run: ${missingDependencies.join(", ")}. Either add them or add the --ignore-missing-stack-dependencies flag.`);
    }
}
exports.checkIfAllDependenciesAreIncluded = checkIfAllDependenciesAreIncluded;
//# sourceMappingURL=data:application/json;base64,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