"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
var __importDefault = (this && this.__importDefault) || function (mod) {
    return (mod && mod.__esModule) ? mod : { "default": mod };
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.PackageManager = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const commons_1 = require("@cdktf/commons");
const fs_extra_1 = require("fs-extra");
const path_1 = __importDefault(require("path"));
const xml_js_1 = require("xml-js");
const fs = __importStar(require("fs-extra"));
const semver = __importStar(require("semver"));
const node_fetch_1 = __importDefault(require("node-fetch"));
const z = __importStar(require("zod"));
// Can't use CDKTF_ as prefix because yargs .env("CDKTF") in strict mode does not allow us to
// Refer to: https://github.com/yargs/yargs/issues/873
const { GITHUB_API_TOKEN_CDKTF } = process.env;
// {
//   "version": "1.0.0",
//   "name": "testUSHasF",
//   "problems": [
//     "extraneous: archiver-utils@2.1.0 /private/var/folders/z_/v03l33d55fb57nrr3b1q03ch0000gq/T/testUSHasF/node_modules/archiver-utils",
//   ],
//   "dependencies": {
//     "@cdktf/provider-random": {
//       "version": "3.0.11",
//       "resolved": "https://registry.npmjs.org/@cdktf/provider-random/-/provider-random-3.0.11.tgz"
//     },
const npmListSchema = z
    .object({
    dependencies: z.record(z
        .object({
        version: z.string(),
    })
        .nonstrict()),
})
    .deepPartial()
    .nonstrict();
// {
//   "type": "tree",
//   "data": {
//     "type": "list",
//     "trees": [
//       {
//         "name": "@cdktf/provider-random@3.0.11",
//         "children": [],
//         "hint": null,
//         "color": "bold",
//         "depth": 0
//       }
//     ]
//   }
// }
const yarnListSchema = z
    .object({
    data: z
        .object({
        trees: z.array(z
            .object({
            name: z.string(),
        })
            .nonstrict()),
    })
        .nonstrict(),
})
    .deepPartial()
    .nonstrict();
// [
//   {
//     "name": "appdirs",
//     "version": "1.4.4"
//   },
//   {
const pipPackageSchema = z.array(z.object({ name: z.string(), version: z.string() }).nonstrict());
/**
 * manages installing, updating, and removing dependencies
 * in the package system used by the target language of a CDKTF
 * project
 */
class PackageManager {
    constructor(workingDirectory) {
        this.workingDirectory = workingDirectory;
    }
    static forLanguage(language, workingDirectory) {
        switch (language) {
            case commons_1.Language.GO:
                return new GoPackageManager(workingDirectory);
            case commons_1.Language.TYPESCRIPT:
                return new NodePackageManager(workingDirectory);
            case commons_1.Language.PYTHON:
                return new PythonPackageManager(workingDirectory);
            case commons_1.Language.CSHARP:
                return new NugetPackageManager(workingDirectory);
            case commons_1.Language.JAVA:
                if (GradlePackageManager.isGradleProject(workingDirectory)) {
                    return new GradlePackageManager(workingDirectory);
                }
                return new MavenPackageManager(workingDirectory);
            default:
                throw new Error(`Unknown language: ${language}`);
        }
    }
}
exports.PackageManager = PackageManager;
class NodePackageManager extends PackageManager {
    hasYarnLockfile() {
        return (0, fs_extra_1.existsSync)(path_1.default.join(this.workingDirectory, "yarn.lock"));
    }
    async addPackage(packageName, packageVersion, silent) {
        console.log(`Adding package ${packageName} @ ${packageVersion}`);
        // probe for package-lock.json or yarn.lock
        let command = "npm";
        let args = ["install"];
        if (this.hasYarnLockfile()) {
            command = "yarn";
            args = ["add"];
        }
        args.push(packageVersion ? packageName + "@" + packageVersion : packageName);
        if (silent) {
            args.push("--silent");
            args.push("--no-progress");
        }
        // Install exact version
        // Yarn: https://classic.yarnpkg.com/lang/en/docs/cli/add/#toc-yarn-add-exact-e
        // Npm: https://docs.npmjs.com/cli/v8/commands/npm-install#save-exact
        args.push("-E");
        commons_1.logger.info(`Installing package ${packageName} @ ${packageVersion} using ${command}.`);
        await (0, commons_1.exec)(command, args, { cwd: this.workingDirectory });
        commons_1.logger.info("Package installed.");
    }
    async isNpmVersionAvailable(_packageName, _packageVersion) {
        // We get the list of available versions from npm, no need to check here
        return true;
    }
    async listYarnPackages() {
        var _a;
        try {
            const stdout = await (0, commons_1.exec)("yarn", ["list", "--json"], {
                cwd: this.workingDirectory,
            });
            commons_1.logger.debug(`Listing yarn packages using "yarn list --json": ${stdout}`);
            const json = yarnListSchema.parse(JSON.parse(stdout));
            return (((_a = json === null || json === void 0 ? void 0 : json.data) === null || _a === void 0 ? void 0 : _a.trees) || [])
                .filter((dep) => dep.name.startsWith("@cdktf/provider-"))
                .map((dep) => ({
                name: `@${dep.name.split("@")[1]}`,
                version: dep.name.split("@")[2],
            }));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages using 'yarn list --json': ${e.message}`);
        }
    }
    async listNpmPackages() {
        try {
            const stdout = await (0, commons_1.exec)("npm", ["list", "--json"], {
                cwd: this.workingDirectory,
            });
            commons_1.logger.debug(`Listing npm packages using "npm list --json": ${stdout}`);
            const json = npmListSchema.parse(JSON.parse(stdout));
            return Object.entries((json === null || json === void 0 ? void 0 : json.dependencies) || {})
                .filter(([depName]) => depName.startsWith("@cdktf/provider-"))
                .map(([name, dep]) => ({ name, version: dep.version }));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages using 'npm list --json': ${e.message}`);
        }
    }
    async listProviderPackages() {
        return this.hasYarnLockfile()
            ? this.listYarnPackages()
            : this.listNpmPackages();
    }
}
class PythonPackageManager extends PackageManager {
    get appCommand() {
        try {
            return JSON.parse(fs.readFileSync(path_1.default.resolve(this.workingDirectory, "cdktf.json"), "utf8"))["app"];
        }
        catch (e) {
            throw commons_1.Errors.Usage(`Could not find find and parse cdktf.json in ${this.workingDirectory}`, e);
        }
    }
    async addPackage(packageName, packageVersion) {
        const usePipenv = this.appCommand.includes("pipenv");
        if (usePipenv) {
            console.log(`Installing package ${packageName} @ ${packageVersion} using pipenv.`);
            await (0, commons_1.exec)("pipenv", ["install", `${packageName}~=${packageVersion}`], {
                cwd: this.workingDirectory,
                env: {
                    ...process.env,
                    PIPENV_QUIET: "1",
                },
                stdio: ["inherit", 1, 1],
            });
            console.log("Package installed.");
        }
        else {
            console.log(`Installing package ${packageName} @ ${packageVersion} using pip.`);
            const requirementsFilePath = path_1.default.join(this.workingDirectory, "requirements.txt");
            if (!fs.existsSync(requirementsFilePath)) {
                throw commons_1.Errors.Usage(`Could not find requirements.txt in ${this.workingDirectory}`);
            }
            const requirements = await fs.readFile(requirementsFilePath, "utf8");
            const requirementLine = requirements
                .split("\n")
                .find((line) => line.includes(packageName));
            commons_1.logger.debug(`Read requirements.txt file and found line including ${packageName}: ${requirementLine}`);
            if (requirementLine) {
                if (packageVersion ? requirementLine.includes(packageVersion) : true) {
                    commons_1.logger.info(`Package ${packageName} already installed. Skipping installation.`);
                    return;
                }
                else {
                    commons_1.logger.debug(`Found the package but with a different version, continuing`);
                }
            }
            const newRequirements = requirements
                .split("\n")
                .filter((line) => !line.startsWith(packageName))
                .join("\n") +
                `\n${packageName}${packageVersion ? `~=${packageVersion}` : ""}`;
            await fs.writeFile(requirementsFilePath, newRequirements, "utf8");
            await (0, commons_1.exec)("pip", ["install", "-r", "requirements.txt"], {
                cwd: this.workingDirectory,
                stdio: ["inherit", 1, 1],
            });
            console.log("Package installed.");
        }
    }
    async isNpmVersionAvailable(packageName, packageVersion) {
        commons_1.logger.debug(`Checking if ${packageName}@${packageVersion} is available for Python`);
        const url = `https://pypi.org/pypi/${packageName}/${packageVersion}/json`;
        commons_1.logger.debug(`Fetching package information for ${packageName} from ${url}`);
        const response = await (0, node_fetch_1.default)(url);
        const json = (await response.json());
        commons_1.logger.debug(`Got response from PyPI for ${packageName}@${packageVersion}: ${JSON.stringify(json)}`);
        if (json.info) {
            // We found the version, so it exists
            return true;
        }
        else {
            commons_1.logger.debug(`Could not get PyPI package info, got: ${JSON.stringify(json)}`);
            return false;
        }
    }
    async listPipenvPackages() {
        try {
            const stdout = await (0, commons_1.exec)("pipenv", ["run", "pip", "list", "--format=json"], {
                cwd: this.workingDirectory,
            });
            commons_1.logger.debug(`Listing pipenv packages using "pipenv run pip list --format=json": ${stdout}`);
            const list = pipPackageSchema.parse(JSON.parse(stdout));
            return list.filter((item) => item.name.startsWith("cdktf-cdktf-provider"));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages using 'pipenv run pip list --format=json': ${e.message}`);
        }
    }
    async listPipPackages() {
        try {
            const stdout = await (0, commons_1.exec)("pip", ["list", "--format=json"], {
                cwd: this.workingDirectory,
            });
            commons_1.logger.debug(`Listing pipenv packages using "pip list --format=json": ${stdout}`);
            const list = pipPackageSchema.parse(JSON.parse(stdout));
            return list.filter((item) => item.name.startsWith("cdktf-cdktf-provider"));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages using 'pip list --format=json': ${e.message}`);
        }
    }
    async listProviderPackages() {
        return this.appCommand.includes("pipenv")
            ? this.listPipenvPackages()
            : this.listPipPackages();
    }
}
class NugetPackageManager extends PackageManager {
    async addPackage(packageName, packageVersion) {
        const command = "dotnet";
        const args = ["add", "package", packageName];
        if (packageVersion) {
            args.push("--version", packageVersion);
        }
        console.log(`Installing package ${packageName} @ ${packageVersion} using "${command} ${args.join(" ")}".`);
        await (0, commons_1.exec)(command, args, { cwd: this.workingDirectory });
        console.log("Package installed.");
    }
    async isNpmVersionAvailable(packageName, packageVersion) {
        var _a, _b, _c;
        commons_1.logger.debug(`Checking if ${packageName}@${packageVersion} is available`);
        const [owner, ...rest] = packageName.split(".");
        const id = rest[rest.length - 1];
        const url = `https://azuresearch-usnc.nuget.org/query?q=owner:${owner}%20id:${id}&prerelease=false&semVerLevel=2.0.0`;
        commons_1.logger.debug(`Fetching package metadata from Nuget: '${url}'`);
        const response = await (0, node_fetch_1.default)(url);
        const json = (await response.json());
        commons_1.logger.debug(`Got response from NuGet for ${packageName} : ${JSON.stringify(json)}`);
        if (!((_a = json === null || json === void 0 ? void 0 : json.data) === null || _a === void 0 ? void 0 : _a.length)) {
            return false; // No package found
        }
        const packageVersions = (_c = (_b = json.data.find((p) => p.id === packageName)) === null || _b === void 0 ? void 0 : _b.versions) !== null && _c !== void 0 ? _c : [];
        if (!packageVersions.length) {
            return false; // No package release matching the id found
        }
        return packageVersions.some((v) => v.version === packageVersion);
    }
    async listProviderPackages() {
        try {
            const stdout = await (0, commons_1.exec)("dotnet", ["list", "package"], {
                cwd: this.workingDirectory,
            });
            commons_1.logger.debug(`Listing pipenv packages using "dotnet list package": ${stdout}`);
            const regex = /^\s*>\s(HashiCorp\.Cdktf\.Providers\.[\w.]+)\s+((?:\d+\.){2}\d+(?:-\S+)?)\s+((?:\d+\.){2}\d+(?:-\S+)?)\s*$/;
            return stdout
                .split("\n")
                .map((line) => {
                // Example output:
                // Project 'MyTerraformStack' has the following package references
                //  [net6.0]:
                //  Top-level Package      Requested   Resolved
                //  > HashiCorp.Cdktf      0.0.0       0.0.0
                // match[0] = full match
                // match[1] = package name
                // match[2] = requested version
                // match[3] = resolved version
                return regex.exec(line);
            })
                .filter((match) => !!match)
                .map((match) => ({ name: match[1], version: match[3] }));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages using 'dotnet list package': ${e.message}`);
        }
    }
}
class JavaPackageManager extends PackageManager {
    async isNpmVersionAvailable(packageName, packageVersion) {
        var _a, _b;
        commons_1.logger.debug(`Checking if ${packageName}@${packageVersion} is available`);
        const parts = packageName.split(".");
        if (parts.length !== 3) {
            throw commons_1.Errors.Internal(`Expected package name to be in format "group.artifact", e.g. "com.hashicorp.cdktf-provider-google", got: ${packageName}`);
        }
        const packageIdentifier = parts.pop();
        const groupId = parts.join(".");
        const url = `https://search.maven.org/solrsearch/select?q=g:${groupId}+AND+a:${packageIdentifier}+AND+v:${packageVersion}&rows=5&wt=json`;
        commons_1.logger.debug(`Trying to find package version by querying Maven Central under '${url}'`);
        const response = await (0, node_fetch_1.default)(url);
        const json = (await response.json());
        commons_1.logger.debug(`Got response from the Maven package search for ${packageName}: ${JSON.stringify(json)}`);
        return ((_b = (_a = json === null || json === void 0 ? void 0 : json.response) === null || _a === void 0 ? void 0 : _a.numFound) !== null && _b !== void 0 ? _b : 0) > 0;
    }
}
class MavenPackageManager extends JavaPackageManager {
    async addPackage(packageName, packageVersion = "LATEST") {
        var _a, _b, _c, _d;
        console.log(`Adding ${packageName} @ ${packageVersion} to pom.xml`);
        // Assert pom.xml exists
        const pomPath = path_1.default.join(this.workingDirectory, "pom.xml");
        if (!(0, fs_extra_1.existsSync)(pomPath)) {
            throw commons_1.Errors.Usage("No pom.xml found in current working directory. Please run the command from the root of your project.");
        }
        const pom = await fs.readFile(pomPath, "utf8");
        const pomXml = (await (0, xml_js_1.xml2js)(pom, {}));
        // Mutate dependencies
        const nameParts = packageName.split(".");
        const groupId = nameParts.slice(0, nameParts.length - 1).join(".");
        const artifactId = nameParts[nameParts.length - 1];
        const newDependency = (await (0, xml_js_1.xml2js)(`<dependency>
    <groupId>${groupId}</groupId>
    <artifactId>${artifactId}</artifactId>
    <version>${packageVersion}</version>
</dependency>`));
        const dependencies = (_c = (_b = (_a = pomXml.elements) === null || _a === void 0 ? void 0 : _a.find((el) => el.name === "project")) === null || _b === void 0 ? void 0 : _b.elements) === null || _c === void 0 ? void 0 : _c.find((el) => el.name === "dependencies");
        if (!dependencies) {
            throw commons_1.Errors.Usage(`Could not find dependencies section in the pom.xml`);
        }
        dependencies.elements = ((dependencies === null || dependencies === void 0 ? void 0 : dependencies.elements) || []).filter((el) => {
            var _a, _b;
            return ((_a = el.elements) === null || _a === void 0 ? void 0 : _a.some((group) => { var _a; return group.name === "groupId" && ((_a = group.elements) === null || _a === void 0 ? void 0 : _a[0].text) !== groupId; })) ||
                ((_b = el.elements) === null || _b === void 0 ? void 0 : _b.some((artifact) => {
                    var _a;
                    return artifact.name === "artifactId" &&
                        ((_a = artifact.elements) === null || _a === void 0 ? void 0 : _a[0].text) !== artifactId;
                }));
        });
        (_d = dependencies === null || dependencies === void 0 ? void 0 : dependencies.elements) === null || _d === void 0 ? void 0 : _d.push(newDependency.elements[0]);
        // Write new pom.xml
        await fs.writeFile(pomPath, (0, xml_js_1.js2xml)(pomXml, { spaces: 2 }));
        // Install
        await (0, commons_1.exec)("mvn", ["install"], { cwd: this.workingDirectory });
        console.log("Package installed.");
    }
    async listProviderPackages() {
        var _a, _b, _c, _d, _e;
        try {
            const pomPath = path_1.default.join(this.workingDirectory, "pom.xml");
            if (!(0, fs_extra_1.existsSync)(pomPath)) {
                throw commons_1.Errors.Usage("No pom.xml found in current working directory. Please run the command from the root of your project.");
            }
            const pom = await fs.readFile(pomPath, "utf8");
            const pomXml = (await (0, xml_js_1.xml2js)(pom, {}));
            const dependencies = (_e = (_d = (_c = (_b = (_a = pomXml.elements) === null || _a === void 0 ? void 0 : _a.find((el) => el.name === "project")) === null || _b === void 0 ? void 0 : _b.elements) === null || _c === void 0 ? void 0 : _c.find((el) => el.name === "dependencies")) === null || _d === void 0 ? void 0 : _d.elements) !== null && _e !== void 0 ? _e : [];
            return dependencies
                .map((dep) => {
                var _a, _b, _c, _d, _e, _f, _g, _h, _j;
                return ({
                    name: `${(_c = (_b = (_a = dep.elements) === null || _a === void 0 ? void 0 : _a.find((el) => el.name === "groupId")) === null || _b === void 0 ? void 0 : _b.elements) === null || _c === void 0 ? void 0 : _c[0].text}.${(_f = (_e = (_d = dep.elements) === null || _d === void 0 ? void 0 : _d.find((el) => el.name === "artifactId")) === null || _e === void 0 ? void 0 : _e.elements) === null || _f === void 0 ? void 0 : _f[0].text}`,
                    version: (_j = (_h = (_g = dep.elements) === null || _g === void 0 ? void 0 : _g.find((el) => el.name === "version")) === null || _h === void 0 ? void 0 : _h.elements) === null || _j === void 0 ? void 0 : _j[0].text,
                });
            })
                .filter((dep) => dep.name.startsWith("com.hashicorp.cdktf-provider-"));
        }
        catch (e) {
            throw new Error(`Could not determine installed packages reading the pom.xml: ${e.message}`);
        }
    }
}
class GradlePackageManager extends JavaPackageManager {
    static isGradleProject(workingDirectory) {
        return (0, commons_1.isGradleProject)(workingDirectory);
    }
    async addPackage(packageFQN, packageVersion = "latest.release") {
        const buildGradlePath = path_1.default.join(this.workingDirectory, "build.gradle");
        const buildGradle = await fs.readFile(buildGradlePath, "utf8");
        const buildGradleLines = buildGradle.split(/\r?\n/);
        const dependenciesRegex = /dependencies\s+\{/i;
        const dependencyBlockStart = buildGradleLines.findIndex((line) => dependenciesRegex.test(line));
        if (dependencyBlockStart === -1) {
            throw commons_1.Errors.Usage("Could not find dependencies section in the build.gradle");
        }
        const packageSegments = packageFQN.split(".");
        const packageName = packageSegments.pop();
        const groupName = packageSegments.join(".");
        const dependencySpecifier = `${groupName}:${packageName}`;
        const dependencyAndVersionSpecifier = `${dependencySpecifier}:${packageVersion}`;
        const existingDependency = buildGradleLines.findIndex((line) => line.includes(dependencySpecifier));
        if (existingDependency !== -1) {
            buildGradleLines.splice(existingDependency, 1);
        }
        const newPackageDependency = `\timplementation '${dependencyAndVersionSpecifier}'`;
        buildGradleLines.splice(dependencyBlockStart + 1, 0, newPackageDependency);
        await fs.writeFile(buildGradlePath, buildGradleLines.join("\n"));
    }
    async listProviderPackages() {
        const dependencies = await (0, commons_1.getGradleDependencies)();
        if (!dependencies) {
            throw commons_1.Errors.Usage("Could not find any dependencies");
        }
        const dependencyList = dependencies
            .map((line) => (0, commons_1.getDependencyInformationFromLine)(line))
            .filter((dep) => {
            if (!dep) {
                return false;
            }
            return dep.name.includes("cdktf-provider-");
        })
            .map((dep) => ({
            name: `com.hashicorp.${dep.name}`,
            version: dep.version,
        }));
        return dependencyList;
    }
}
class GoPackageManager extends PackageManager {
    async addPackage(packageName, packageVersion) {
        console.log(`Adding package ${packageName} @ ${packageVersion}`);
        const majorVersion = packageVersion
            ? semver.major(packageVersion)
            : undefined;
        let versionPackageSuffix = "";
        if (typeof majorVersion === "number" && majorVersion > 1) {
            versionPackageSuffix = `/v${majorVersion}`;
        }
        commons_1.logger.debug(`Running 'go get ${packageName}${versionPackageSuffix}@v${packageVersion}'`);
        // Install
        await (0, commons_1.exec)("go", ["get", `${packageName}${versionPackageSuffix}@v${packageVersion}`], {
            cwd: this.workingDirectory,
        });
        console.log("Package installed.");
    }
    async isNpmVersionAvailable(packageName, packageVersion) {
        commons_1.logger.debug(`Checking if ${packageName}@${packageVersion} is available`);
        // e.g. github.com/cdktf/cdktf-provider-google-go/google
        const parts = packageName.split("/");
        if (parts.length !== 4) {
            throw commons_1.Errors.Internal(`Expecting Go package name to be in the format of github.com/<org>/<repo>/<package>, got ${packageName}`);
        }
        const org = parts[1];
        const repo = parts[2];
        const packagePath = parts[3];
        const url = `https://api.github.com/repos/${org}/${repo}/git/ref/tags/${packagePath}/v${packageVersion}`;
        commons_1.logger.debug(`Fetching tags for ${org}/${repo} from '${url}'`);
        const response = await (0, node_fetch_1.default)(url, {
            headers: {
                Accept: "application/vnd.github+json",
                "User-Agent": "HashiCorp/cdktf-cli",
                ...(GITHUB_API_TOKEN_CDKTF
                    ? { Authorization: `Bearer ${GITHUB_API_TOKEN_CDKTF}` }
                    : {}),
            },
        });
        const json = (await response.json());
        commons_1.logger.debug(`Got response from GitHubs repository tag endpoint for ${packageName}: ${JSON.stringify(json)}`);
        if (json && json.ref) {
            return true;
        }
        commons_1.logger.info(`Could not find the tag ${packagePath}/v${packageVersion} in the repository ${org}/${repo}: ${JSON.stringify(json)}}`);
        return false;
    }
    async listProviderPackages() {
        try {
            const goSumPath = path_1.default.join(this.workingDirectory, "go.sum");
            if (!(0, fs_extra_1.existsSync)(goSumPath)) {
                throw commons_1.Errors.Usage("No go.sum found in current working directory. Please run the command from the root of your project.");
            }
            const goSum = await fs.readFile(goSumPath, "utf8");
            const dedupedProviderNames = new Set();
            return goSum
                .split("\n")
                .filter((line) => line.startsWith("github.com/hashicorp/cdktf-provider") ||
                line.startsWith("github.com/cdktf/cdktf-provider"))
                .map((line) => {
                const parts = line.split(" ");
                if (parts.length !== 3) {
                    throw commons_1.Errors.Internal(`Expected line in go.sum to be in the format of '<package> <version> <checksum>', got: ${line}`);
                }
                // part[0] could be github.com/aws/constructs-go/constructs/v10
                const name = parts[0].split("/").slice(0, 4).join("/");
                const version = parts[1].split("/")[0];
                if (dedupedProviderNames.has(name)) {
                    return { name: "", version: "" };
                }
                dedupedProviderNames.add(name);
                return {
                    name,
                    version,
                };
            })
                .filter((providerInfo) => !!providerInfo.name && !!providerInfo.version);
        }
        catch (e) {
            throw new Error(`Could not determine installed packages reading the go.sum: ${e.message}`);
        }
    }
}
//# sourceMappingURL=data:application/json;base64,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