"use strict";
var __createBinding = (this && this.__createBinding) || (Object.create ? (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    var desc = Object.getOwnPropertyDescriptor(m, k);
    if (!desc || ("get" in desc ? !m.__esModule : desc.writable || desc.configurable)) {
      desc = { enumerable: true, get: function() { return m[k]; } };
    }
    Object.defineProperty(o, k2, desc);
}) : (function(o, m, k, k2) {
    if (k2 === undefined) k2 = k;
    o[k2] = m[k];
}));
var __setModuleDefault = (this && this.__setModuleDefault) || (Object.create ? (function(o, v) {
    Object.defineProperty(o, "default", { enumerable: true, value: v });
}) : function(o, v) {
    o["default"] = v;
});
var __importStar = (this && this.__importStar) || function (mod) {
    if (mod && mod.__esModule) return mod;
    var result = {};
    if (mod != null) for (var k in mod) if (k !== "default" && Object.prototype.hasOwnProperty.call(mod, k)) __createBinding(result, mod, k);
    __setModuleDefault(result, mod);
    return result;
};
Object.defineProperty(exports, "__esModule", { value: true });
exports.DependencyManager = exports.ProviderConstraint = exports.DEFAULT_NAMESPACE = exports.DEFAULT_HOSTNAME = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const commons_1 = require("@cdktf/commons");
const codemaker_1 = require("codemaker");
const cdktf_config_1 = require("../cdktf-config");
const cdktf_config_manager_1 = require("./cdktf-config-manager");
const package_manager_1 = require("./package-manager");
const prebuilt_providers_1 = require("./prebuilt-providers");
const registry_api_1 = require("./registry-api");
const version_constraints_1 = require("./version-constraints");
const semver = __importStar(require("semver"));
const local_provider_versions_1 = require("../local-provider-versions");
const local_provider_constraints_1 = require("../local-provider-constraints");
// ref: https://www.terraform.io/language/providers/requirements#source-addresses
exports.DEFAULT_HOSTNAME = "registry.terraform.io";
exports.DEFAULT_NAMESPACE = "hashicorp";
function normalizeProviderSource(source) {
    // returns <HOSTNAME>/<NAMESPACE>/<TYPE>
    const slashes = source.split("/").length - 1;
    switch (slashes) {
        case 0:
            return `${exports.DEFAULT_HOSTNAME}/${exports.DEFAULT_NAMESPACE}/${source}`;
        case 1:
            return `${exports.DEFAULT_HOSTNAME}/${source}`;
        default:
            return source;
    }
}
class ProviderConstraint {
    // TODO: add examples to cli command description (i.e. =,~>.> etc.)
    // if no version constraint is specified, we assume the latest version
    // if specific version is specified without e.g. =, we allow patch level increments (e.g. ~>2.12 for "2.12")
    constructor(source, version) {
        this.version = version;
        this.source = normalizeProviderSource(source);
    }
    static fromConfigEntry(provider) {
        if (typeof provider === "string") {
            const [src, version] = provider.split("@");
            return new ProviderConstraint(src.trim(), version ? version.trim() : undefined);
        }
        const src = (provider.namespace ? `${provider.namespace}/` : "") +
            (provider.source || provider.name);
        return new ProviderConstraint(src, provider.version);
    }
    isFromTerraformRegistry() {
        return this.hostname === exports.DEFAULT_HOSTNAME;
    }
    /**
     * the namespace of the provider
     * e.g. "hashicorp" or "kreuzwerker"
     */
    get namespace() {
        return this.source.split("/")[1];
    }
    /**
     * the name of the provider
     * e.g. "aws"
     */
    get name() {
        return this.source.split("/")[2];
    }
    /**
     * the hostname of the provider
     * e.g. "registry.terraform.io"
     */
    get hostname() {
        return this.source.split("/")[0];
    }
    /**
     * returns a simplified provider name, dropping namespace and hostname
     * if they match the defaults
     */
    get simplifiedName() {
        return this.source
            .split("/")
            .filter((part) => part !== exports.DEFAULT_HOSTNAME && part !== exports.DEFAULT_NAMESPACE)
            .join("/");
    }
    /**
     * checks if the version constraint matches the given version
     * @param version an actual version (e.g. "4.12.1")
     * @returns true if the version is compatible with the constraint
     */
    matchesVersion(version) {
        if (this.version) {
            return (0, version_constraints_1.versionMatchesConstraint)(version, this.version);
        }
        return true;
    }
    toString() {
        return `${this.source}${this.version ? `@${this.version}` : ""}`;
    }
}
exports.ProviderConstraint = ProviderConstraint;
/**
 * manages dependencies of a CDKTF project (e.g. terraform providers)
 */
class DependencyManager {
    constructor(targetLanguage, cdktfVersion, projectDirectory) {
        this.targetLanguage = targetLanguage;
        this.cdktfVersion = cdktfVersion;
        this.projectDirectory = projectDirectory;
        this.packageManager = package_manager_1.PackageManager.forLanguage(targetLanguage, this.projectDirectory);
    }
    async addProvider(constraint) {
        if (await this.hasPrebuiltProvider(constraint)) {
            await this.addPrebuiltProvider(constraint);
            return { addedLocalProvider: false };
        }
        else {
            await this.addLocalProvider(constraint);
            return { addedLocalProvider: true };
        }
    }
    async upgradeProvider(constraint) {
        console.log(`Upgrading ${constraint.simplifiedName}...`);
        const cdktfJson = new cdktf_config_manager_1.CdktfConfigManager();
        const prebuiltVersion = await this.getCurrentlyInstalledVersion(constraint);
        if (prebuiltVersion) {
            await this.upgradePrebuiltProvider(constraint, prebuiltVersion);
            return { addedLocalProvider: false };
        }
        if (await cdktfJson.hasProvider(constraint)) {
            await cdktfJson.updateProvider(constraint);
            return { addedLocalProvider: true };
        }
        throw commons_1.Errors.Usage(`Trying to upgrade ${constraint.simplifiedName} but it is not installed, please use "cdktf provider add ${constraint.simplifiedName}" to add it.`);
    }
    async getCurrentlyInstalledVersion(constraint) {
        var _a;
        commons_1.logger.info(`Checking if ${constraint.simplifiedName} is installed...`);
        const packageName = await this.tryGetPackageName(constraint);
        if (!packageName)
            return; // not available as pre-built provider, so can't be installed as such
        commons_1.logger.debug(`Expecting package ${packageName} to be installed if provider is installed as pre-built one`);
        let installedPackages;
        try {
            installedPackages = await this.packageManager.listProviderPackages();
        }
        catch (e) {
            throw new Error(`Failed to list packages: ${e}`);
        }
        commons_1.logger.debug(`Installed packages found: ${JSON.stringify(installedPackages, null, 2)}`);
        return (_a = installedPackages.find((pkg) => pkg.name === packageName)) === null || _a === void 0 ? void 0 : _a.version;
    }
    async upgradePrebuiltProvider(constraint, currentVersion) {
        commons_1.logger.debug(`Searching for latest matching version of ${constraint.simplifiedName}`);
        const packageName = await this.getPackageName(constraint);
        const packageVersion = await this.getMatchingProviderVersion(constraint);
        commons_1.logger.debug(`Found package ${packageName}@${packageVersion}`);
        if (packageVersion !== currentVersion) {
            await this.packageManager.addPackage(packageName, packageVersion);
        }
        else {
            console.log(`The latest version of ${packageName} is already installed: ${packageVersion}`);
        }
    }
    async hasPrebuiltProvider(constraint) {
        commons_1.logger.debug(`determining whether pre-built provider exists for ${constraint.source} with version constraint ${constraint.version} and cdktf version ${this.cdktfVersion}`);
        commons_1.logger.info(`Checking whether pre-built provider exists for the following constraints:
  provider: ${constraint.simplifiedName}
  version : ${constraint.version || "latest"}
  language: ${this.targetLanguage}
  cdktf   : ${this.cdktfVersion}
`);
        if (this.targetLanguage === commons_1.Language.GO &&
            semver.lt(this.cdktfVersion, "0.12.0")) {
            commons_1.logger.info(`Before CDKTF 0.12.0 there were no pre-built providers published for Go.`);
            return false;
        }
        const v = await (0, prebuilt_providers_1.getPrebuiltProviderVersions)(constraint, this.cdktfVersion);
        const exists = v !== null;
        if (exists) {
            commons_1.logger.info(`Found pre-built provider.`);
        }
        else {
            commons_1.logger.info(`Pre-built provider does not exist for the given constraints.`);
        }
        return exists;
    }
    async tryGetPackageName(constraint) {
        const npmPackageName = await (0, prebuilt_providers_1.getNpmPackageName)(constraint);
        if (!npmPackageName)
            return;
        const prebuiltProviderRepository = await (0, prebuilt_providers_1.getPrebuiltProviderRepositoryName)(npmPackageName);
        return this.convertPackageName(npmPackageName, prebuiltProviderRepository);
    }
    async getPackageName(constraint) {
        const packageName = await this.tryGetPackageName(constraint);
        if (!packageName) {
            throw commons_1.Errors.Usage(`Could not find pre-built provider for ${constraint.source}`);
        }
        return packageName;
    }
    async getMatchingProviderVersion(constraint) {
        const packageName = await this.getPackageName(constraint);
        const prebuiltProviderNpmVersions = await (0, prebuilt_providers_1.getPrebuiltProviderVersions)(constraint, this.cdktfVersion);
        if (!prebuiltProviderNpmVersions) {
            throw commons_1.Errors.Usage(`No pre-built provider found for ${constraint.source} with version constraint ${constraint.version} and cdktf version ${this.cdktfVersion}`);
        }
        const packageVersion = await this.getLanguageSpecificPackageVersion(packageName, prebuiltProviderNpmVersions);
        if (!packageVersion) {
            throw commons_1.Errors.Usage(`No pre-built provider found for ${constraint.source} with version constraint ${constraint.version} and cdktf version ${this.cdktfVersion} for language ${this.targetLanguage}.`);
        }
        return packageVersion;
    }
    async addPrebuiltProvider(constraint, silent = false) {
        commons_1.logger.debug(`adding pre-built provider ${constraint.source} with version constraint ${constraint.version} for cdktf version ${this.cdktfVersion}`);
        const packageName = await this.getPackageName(constraint);
        const packageVersion = await this.getMatchingProviderVersion(constraint);
        await this.packageManager.addPackage(packageName, packageVersion, silent);
        // TODO: more debug logs
    }
    // The version we use for npm might differ from other registries
    // This happens mostly in cases where a provider update failed to publish to one of the registries
    // In that case we use the latest version that was published successfully and works with the current cdktf release
    async getLanguageSpecificPackageVersion(packageName, prebuiltProviderNpmVersions) {
        commons_1.logger.debug("Found possibly matching versions (released on npm): ", prebuiltProviderNpmVersions);
        commons_1.logger.debug("Searching through package manager to find latest available version for given language");
        for (const version of prebuiltProviderNpmVersions) {
            try {
                const isAvailable = await this.packageManager.isNpmVersionAvailable(packageName, version);
                if (isAvailable) {
                    return version;
                }
            }
            catch (err) {
                commons_1.logger.info(`Could not find version ${version} for package ${packageName}: '${err}'. Skipping...`);
            }
        }
        return null;
    }
    async addLocalProvider(constraint) {
        commons_1.logger.info(`Adding local provider ${constraint.source} with version constraint ${constraint.version} to cdktf.json`);
        if (!constraint.version && constraint.isFromTerraformRegistry()) {
            const v = await (0, registry_api_1.getLatestVersion)(constraint);
            if (v) {
                constraint = new ProviderConstraint(constraint.source, 
                // "1.3.2" -> "~> 1.3"
                `~> ${v.split(".").slice(0, 2).join(".")}`);
            }
            else {
                throw commons_1.Errors.Usage(`Could not find a version for the provider '${constraint}' in the public registry. This could be due to a typo, please take a look at https://cdk.tf/registry-providers to find all supported providers.`);
            }
        }
        await new cdktf_config_manager_1.CdktfConfigManager().addProvider(constraint);
    }
    /**
     * Converts an NPM package name of a pre-built provider package to the name in the target language
     */
    convertPackageName(name, repository) {
        const providerName = name.replace("@cdktf/provider-", "");
        switch (this.targetLanguage) {
            case commons_1.Language.GO: // e.g. github.com/cdktf/cdktf-provider-opentelekomcloud-go/opentelekomcloud
                if (repository) {
                    return `${repository}-go/${providerName}`;
                }
                return `github.com/cdktf/cdktf-provider-${providerName}-go/${providerName}`;
            case commons_1.Language.TYPESCRIPT: // e.g. @cdktf/provider-random
                return name; // already the correct name
            case commons_1.Language.CSHARP: // e.g. HashiCorp.Cdktf.Providers.Opentelekomcloud
                return `HashiCorp.Cdktf.Providers.` + (0, codemaker_1.toPascalCase)(providerName);
            case commons_1.Language.JAVA: // e.g. com.hashicorp.opentelekomcloud
                return `com.hashicorp.cdktf-provider-${providerName}`;
            case commons_1.Language.PYTHON: // e.g. cdktf-cdktf-provider-opentelekomcloud
                return `cdktf-cdktf-provider-${providerName}`;
            default:
                throw new Error(`converting package name for language ${this.targetLanguage} not implemented yet`);
        }
    }
    /**
     * Converts an package name of a pre-built provider package in target language to the name in npm
     * Inverse of: `convertPackageName`
     */
    convertFromPackageNameToNpm(name) {
        const npmPackagePrefix = "@cdktf/provider-";
        const regexes = {
            [commons_1.Language.GO]: /github.com\/(?:cdktf|hashicorp)\/cdktf-provider-(.+)-go\//i,
            [commons_1.Language.TYPESCRIPT]: /(.+)/i,
            [commons_1.Language.CSHARP]: /HashiCorp\.Cdktf\.Providers\.(.+)/i,
            [commons_1.Language.JAVA]: /com\.hashicorp\.cdktf-provider-(.+)/i,
            [commons_1.Language.PYTHON]: /cdktf-cdktf-provider-(.+)/i,
        };
        const regex = regexes[this.targetLanguage];
        if (!regex) {
            throw commons_1.Errors.Usage("Language not supported for pre-built providers");
        }
        const match = regex.exec(name);
        if (!match) {
            throw new Error(`Package name is not in expected format: ${name}`);
        }
        switch (this.targetLanguage) {
            case commons_1.Language.GO: // e.g. github.com/cdktf/cdktf-provider-opentelekomcloud-go/opentelekomcloud
                return npmPackagePrefix + match[1];
            case commons_1.Language.TYPESCRIPT: // e.g. @cdktf/provider-random
                return match[1]; // already the correct name
            case commons_1.Language.CSHARP: // e.g. HashiCorp.Cdktf.Providers.Opentelekomcloud
                return npmPackagePrefix + (0, codemaker_1.toSnakeCase)(match[1]);
            case commons_1.Language.JAVA: // e.g. com.hashicorp.opentelekomcloud
                return npmPackagePrefix + match[1];
            case commons_1.Language.PYTHON: // e.g. cdktf-cdktf-provider-opentelekomcloud
                return npmPackagePrefix + match[1];
            default:
                throw new Error(`converting package name for language ${this.targetLanguage} not implemented yet`);
        }
    }
    async allProviders() {
        const cdktfJson = cdktf_config_1.CdktfConfig.read();
        const localVersions = new local_provider_versions_1.LocalProviderVersions();
        const localProviderConfigs = cdktfJson.terraformProviders;
        const prebuiltProviderConfigs = await this.packageManager.listProviderPackages();
        const prebuiltProvidersInfo = await Promise.all(prebuiltProviderConfigs.map(async (prebuiltProviderConfig) => {
            const packageName = this.convertFromPackageNameToNpm(prebuiltProviderConfig.name);
            const providerInformation = await (0, prebuilt_providers_1.getPrebuiltProviderVersionInformation)(packageName, prebuiltProviderConfig.version);
            return {
                ...providerInformation,
                packageName: prebuiltProviderConfig.name,
            };
        }));
        const constraints = new local_provider_constraints_1.LocalProviderConstraints();
        const localProvidersInfo = await Promise.all(localProviderConfigs.map(async (localProviderConfig) => {
            const constraint = ProviderConstraint.fromConfigEntry(localProviderConfig);
            const version = await localVersions.versionForProvider(constraint.simplifiedName);
            const constraintValue = await constraints.constraintForProvider(constraint.simplifiedName);
            return {
                providerName: constraint.simplifiedName,
                providerConstraint: constraintValue || constraint.version,
                providerVersion: version,
            };
        }));
        return {
            local: localProvidersInfo,
            prebuilt: prebuiltProvidersInfo,
        };
    }
}
exports.DependencyManager = DependencyManager;
//# sourceMappingURL=data:application/json;base64,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