"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfStack = void 0;
const output_1 = require("./output");
const commons_1 = require("@cdktf/commons");
const terraform_logs_1 = require("./server/terraform-logs");
const terraform_cli_1 = require("./models/terraform-cli");
const dependency_manager_1 = require("./dependencies/dependency-manager");
const terraform_json_1 = require("./terraform-json");
const terraform_provider_lock_1 = require("./terraform-provider-lock");
const convert_1 = require("./convert");
async function getTerraformClient(abortSignal, stack, createTerraformLogHandler) {
    return new terraform_cli_1.TerraformCli(abortSignal, stack, createTerraformLogHandler);
}
class CdktfStack {
    constructor(options) {
        this.options = options;
        this.stopped = false;
        this.currentState = "idle";
        this.stack = options.stack;
        this.parsedContent = terraform_json_1.terraformJsonSchema.parse(JSON.parse(this.stack.content));
    }
    get isPending() {
        return this.currentState === "idle" && !this.stopped;
    }
    get isDone() {
        return (this.currentState === "done" ||
            this.currentState === "errored" ||
            this.stopped);
    }
    get isRunning() {
        return !this.isPending && !this.isDone;
    }
    updateState(update) {
        commons_1.logger.debug(`[${this.stack.name}]: ${update.type}`);
        this.currentState = update.type;
        switch (update.type) {
            case "idle":
            case "done":
                break;
            case "errored":
                this.error = update.error;
                this.options.onUpdate(update);
                break;
            case "outputs fetched":
            case "deployed":
                commons_1.logger.debug(`Outputs: ${JSON.stringify(update.outputs)}`);
                commons_1.logger.debug(`OutputsByConstructId: ${JSON.stringify(update.outputsByConstructId)}`);
                this.outputs = update.outputs;
                this.outputsByConstructId = update.outputsByConstructId;
                this.options.onUpdate(update);
                break;
            default:
                this.options.onUpdate(update);
                break;
        }
    }
    createTerraformLogHandler(phase, filters) {
        commons_1.logger.debug("Creating terraform log handler", phase);
        const onLog = this.options.onLog;
        return (msg, isError = false) => {
            const message = (0, terraform_logs_1.extractJsonLogIfPresent)(msg);
            commons_1.logger.debug(`[${this.options.stack.name}](${phase}): ${msg}`);
            const filterToApply = filters === null || filters === void 0 ? void 0 : filters.find((filter) => filter.condition(message));
            const filteredMessage = filterToApply
                ? filterToApply.transform(message)
                : message;
            if (filteredMessage) {
                commons_1.logger.debug(`Filter ${filterToApply} applied on line '${message}' with result '${filteredMessage}'`);
            }
            if (onLog) {
                onLog({ message: filteredMessage, isError });
            }
        };
    }
    async terraformClient() {
        return await getTerraformClient(this.options.abortSignal, this.options.stack, this.createTerraformLogHandler.bind(this));
    }
    async initalizeTerraform(noColor, skipProviderLock, migrateState) {
        const terraform = await this.terraformClient();
        const needsLockfileUpdate = skipProviderLock
            ? false
            : await this.checkNeedsLockfileUpdate();
        const needsUpgrade = await this.checkNeedsUpgrade();
        await terraform.init({
            needsUpgrade,
            noColor: noColor !== null && noColor !== void 0 ? noColor : false,
            needsLockfileUpdate,
            migrateState: migrateState !== null && migrateState !== void 0 ? migrateState : false,
        });
        return terraform;
    }
    requiredProviders() {
        var _a;
        // Read required providers from the stack output
        const requiredProviders = (_a = this.parsedContent.terraform) === null || _a === void 0 ? void 0 : _a.required_providers;
        return Object.values(requiredProviders || {}).reduce((acc, obj) => {
            const constraint = new dependency_manager_1.ProviderConstraint(obj.source, obj.version);
            acc[constraint.source] = constraint;
            return acc;
        }, {});
    }
    async checkNeedsLockfileUpdate() {
        if (this.options.migrateState) {
            // If we're migrating state, we need to init
            return true;
        }
        const lock = new terraform_provider_lock_1.TerraformProviderLock(this.stack.workingDirectory);
        const lockFileExists = await lock.hasProviderLockFile();
        if (!lockFileExists) {
            // If we don't have a lock file, this is probably the first init
            return true;
        }
        const requiredProviders = this.requiredProviders();
        for (const provider of Object.values(requiredProviders)) {
            const hasProvider = await lock.hasMatchingProvider(provider);
            if (!hasProvider) {
                // If we don't have a provider or version doesn't match, we need to init
                return true;
            }
        }
        return false;
    }
    async checkNeedsUpgrade() {
        const lock = new terraform_provider_lock_1.TerraformProviderLock(this.stack.workingDirectory);
        const allProviders = this.requiredProviders();
        const lockedProviders = Object.values(await lock.providers());
        // Check if any provider contained in `providers` violates constraints in `lockedProviders`
        // Upgrade if some provider constraint not met
        // If a provider wasn't preset in lockedProviders, that's fine; it will just get added
        return lockedProviders.some((lockedProvider) => {
            var _a;
            const lockedConstraint = lockedProvider.constraints;
            if (!lockedConstraint) {
                // Provider lock doesn't have a constraint specified, so we can't check.
                // This shouldn't happen
                commons_1.logger.debug(`Provider lock doesn't have a constraint for ${lockedProvider.name}`);
                return false;
            }
            const provider = allProviders[lockedConstraint.source];
            if (!provider) {
                // else no longer using this provider, so won't cause problems
                return;
            }
            return !lockedConstraint.matchesVersion((_a = provider.version) !== null && _a !== void 0 ? _a : ">0");
        });
    }
    async run(cb) {
        if (this.stopped) {
            return;
        }
        try {
            this.currentWorkPromise = cb();
            await this.currentWorkPromise;
            this.updateState({ type: "done" });
        }
        catch (e) {
            commons_1.logger.trace("Error in currentWorkPromise", e);
            this.currentWorkPromise = undefined;
            this.updateState({
                type: "errored",
                stackName: this.stack.name,
                error: String(e),
            });
            throw e;
        }
        finally {
            this.currentWorkPromise = undefined;
        }
    }
    async diff({ refreshOnly, terraformParallelism, vars, varFiles, noColor, }) {
        await this.run(async () => {
            this.updateState({ type: "planning", stackName: this.stack.name });
            const terraform = await this.terraformClient();
            await terraform.plan({
                destroy: false,
                refreshOnly,
                parallelism: terraformParallelism,
                vars,
                varFiles,
                noColor,
            });
            this.updateState({ type: "planned", stackName: this.stack.name });
            // Find generated file
            const configFile = await (0, terraform_cli_1.tryReadGeneratedConfigurationFile)(this.stack.workingDirectory);
            if (configFile) {
                this.updateState({
                    type: "import with configuration detected",
                    stackName: this.stack.name,
                    configuration: configFile,
                });
                const convertedCode = await (0, convert_1.convertConfigurationFile)(configFile, this.stack.workingDirectory);
                this.updateState({
                    type: "import with configuration converted",
                    stackName: this.stack.name,
                    configuration: convertedCode,
                });
                const onLog = this.options.onLog;
                if (onLog) {
                    onLog({
                        message: `Import without configuration detected. Terraform has created configuration for it:
${configFile}

CDKTF has translated the code to the following:

${convertedCode}

Please review the code and make any necessary changes before adding it to your codebase.
Make sure to only copy the code within the construct's constructor.

NOTE: Your resource has not yet become managed by CDKTF. 
To finish the import remove the call "generateConfigForImport", add the above code within the construct's constructor, and then append the call importFrom(<resource_id_to_import_from>) to the generated code: 

new SomeResource(...).importFrom("some_id")
`,
                        isError: false,
                    });
                }
                await (0, terraform_cli_1.tryRemoveGeneratedConfigurationFile)(this.stack.workingDirectory);
            }
        });
    }
    async deploy(opts) {
        const { refreshOnly, terraformParallelism, noColor, vars, varFiles } = opts;
        await this.run(async () => {
            this.updateState({ type: "planning", stackName: this.stack.name });
            const terraform = await this.terraformClient();
            const { cancelled } = await terraform.deploy({
                autoApprove: this.options.autoApprove,
                refreshOnly,
                parallelism: terraformParallelism,
                vars,
                varFiles,
                noColor,
            }, (state) => {
                // state updates while apply runs that affect the UI
                if (state.type === "running" && !state.cancelled) {
                    this.updateState({
                        type: "deploying",
                        stackName: this.stack.name,
                    });
                }
                else if (state.type === "waiting for approval") {
                    this.updateState({
                        type: "waiting for stack approval",
                        stackName: this.stack.name,
                        approve: state.approve,
                        reject: () => {
                            state.reject();
                            this.updateState({
                                type: "dismissed",
                                stackName: this.stack.name,
                            });
                        },
                    });
                }
                else if (state.type === "waiting for sentinel override") {
                    this.updateState({
                        type: "waiting for stack sentinel override",
                        stackName: this.stack.name,
                        override: state.override,
                        reject: () => {
                            state.reject();
                            this.updateState({
                                type: "dismissed",
                                stackName: this.stack.name,
                            });
                        },
                    });
                }
                else if (state.type === "external approval reply") {
                    this.updateState({
                        type: "external stack approval reply",
                        stackName: this.stack.name,
                        approved: state.approved,
                    });
                }
                else if (state.type === "external sentinel override reply") {
                    this.updateState({
                        type: "external stack sentinel override reply",
                        stackName: this.stack.name,
                        overridden: state.overridden,
                    });
                }
            });
            if (!cancelled) {
                const outputs = await terraform.output();
                const outputsByConstructId = (0, output_1.getConstructIdsForOutputs)(JSON.parse(this.stack.content), outputs);
                this.updateState({
                    type: "deployed",
                    stackName: this.stack.name,
                    outputs,
                    outputsByConstructId,
                });
            }
        });
    }
    async destroy(opts) {
        const { terraformParallelism, noColor, vars, varFiles } = opts;
        await this.run(async () => {
            this.updateState({ type: "planning", stackName: this.stack.name });
            const terraform = await this.terraformClient();
            const { cancelled } = await terraform.destroy({
                autoApprove: this.options.autoApprove,
                parallelism: terraformParallelism,
                vars,
                varFiles,
                noColor,
            }, (state) => {
                // state updates while apply runs that affect the UI
                if (state.type === "running" && !state.cancelled) {
                    this.updateState({
                        type: "destroying",
                        stackName: this.stack.name,
                    });
                }
                else if (state.type === "waiting for approval") {
                    this.updateState({
                        type: "waiting for stack approval",
                        stackName: this.stack.name,
                        approve: state.approve,
                        reject: () => {
                            state.reject();
                            this.updateState({
                                type: "dismissed",
                                stackName: this.stack.name,
                            });
                        },
                    });
                }
                else if (state.type === "waiting for sentinel override") {
                    this.updateState({
                        type: "waiting for stack sentinel override",
                        stackName: this.stack.name,
                        override: state.override,
                        reject: () => {
                            state.reject();
                            this.updateState({
                                type: "dismissed",
                                stackName: this.stack.name,
                            });
                        },
                    });
                }
                else if (state.type === "external approval reply") {
                    this.updateState({
                        type: "external stack approval reply",
                        stackName: this.stack.name,
                        approved: state.approved,
                    });
                }
                else if (state.type === "external sentinel override reply") {
                    this.updateState({
                        type: "external stack sentinel override reply",
                        stackName: this.stack.name,
                        overridden: state.overridden,
                    });
                }
            });
            if (!cancelled)
                this.updateState({
                    type: "destroyed",
                    stackName: this.stack.name,
                });
        });
    }
    async fetchOutputs() {
        await this.run(async () => {
            const terraform = await this.terraformClient();
            const outputs = await terraform.output();
            const outputsByConstructId = (0, output_1.getConstructIdsForOutputs)(JSON.parse(this.stack.content), outputs);
            this.updateState({
                type: "outputs fetched",
                stackName: this.stack.name,
                outputs,
                outputsByConstructId,
            });
        });
        return this.outputs;
    }
    async stop() {
        this.stopped = true;
    }
}
exports.CdktfStack = CdktfStack;
//# sourceMappingURL=data:application/json;base64,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