"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfProject = exports.isWaitingForUserInputUpdate = void 0;
// Copyright (c) HashiCorp, Inc
// SPDX-License-Identifier: MPL-2.0
const commons_1 = require("@cdktf/commons");
const synth_stack_1 = require("./synth-stack");
const synth_1 = require("./synth");
const cdktf_stack_1 = require("./cdktf-stack");
const execution_logs_1 = require("./execution-logs");
const stack_helpers_1 = require("./helpers/stack-helpers");
const cdktf_project_io_handler_1 = require("./cdktf-project-io-handler");
function isWaitingForUserInputUpdate(update) {
    return ["waiting for approval", "waiting for sentinel override"].includes(update.type);
}
exports.isWaitingForUserInputUpdate = isWaitingForUserInputUpdate;
class CdktfProject {
    constructor({ synthCommand, outDir, onUpdate, onLog, workingDirectory = process.cwd(), synthOrigin, hcl = false, }) {
        // Set during deploy / destroy
        this.stacksToRun = [];
        // This means sth different in deploy / destroy
        this.stopAllStacksThatCanNotRunWithout = () => { }; // eslint-disable-line @typescript-eslint/no-empty-function
        this.synthCommand = synthCommand;
        this.outDir = outDir;
        this.workingDirectory = workingDirectory;
        this.onUpdate = onUpdate;
        this.onLog = onLog;
        const ac = new AbortController();
        this.abortSignal = ac.signal;
        this.synthOrigin = synthOrigin;
        this.hcl =
            process.env.SYNTH_HCL_OUTPUT === "true" ||
                process.env.SYNTH_HCL_OUTPUT === "1" ||
                hcl;
        this.hardAbort = ac.abort.bind(ac);
        this.ioHandler = new cdktf_project_io_handler_1.CdktfProjectIOHandler();
    }
    stopAllStacks() {
        this.stacksToRun.forEach((stack) => stack.stop());
        this.ioHandler.filterUserInputEventsFromBuffer();
    }
    handleUserUpdate(update, operations, originalCallback, eventType) {
        const callbacks = (update) => Object.fromEntries(Object.entries(operations).map(([key, value]) => {
            return [
                key,
                // This is passed in to make typescript happy only
                // eslint-disable-next-line @typescript-eslint/no-unused-vars
                (_) => {
                    value(update);
                    // We need to defer these calls for the case that approve() is instantly invoked
                    // in the listener that receives these callbacks as it otherwise would already
                    // remove the "waiting for stack approval" event from the buffer before we even
                    // set waitingForApproval to true (at the end of this if statement) which results
                    // in buffered updates which will never unblock
                    setTimeout(() => this.ioHandler.resumeAfterUserInput(update.stackName), 0);
                },
            ];
        }));
        // always send to buffer, as resumeAfterUserInput() always expects a matching event
        this.ioHandler.pushEvent({
            cb: originalCallback,
            value: {
                type: eventType,
                stackName: update.stackName,
                ...callbacks(update),
            },
            type: "projectUpdate",
        });
        // if we aren't already waiting, this needs to go to cb() too to arrive at the UI
        if (!this.ioHandler.isWaitingForUserInput()) {
            originalCallback({
                type: eventType,
                stackName: update.stackName,
                ...callbacks(update),
            });
        }
    }
    handleUserInputProcess(cb) {
        return (update) => {
            if (update.type === "external stack approval reply") {
                if (!update.approved) {
                    this.stopAllStacksThatCanNotRunWithout(update.stackName);
                }
                this.ioHandler.resumeAfterUserInput(update.stackName);
                return; // aka don't send this event to any buffer
            }
            if (update.type === "external stack sentinel override reply") {
                if (!update.overridden) {
                    this.stopAllStacksThatCanNotRunWithout(update.stackName);
                }
                this.ioHandler.resumeAfterUserInput(update.stackName);
                return; // aka don't send this event to any buffer
            }
            if (update.type === "waiting for stack approval" ||
                update.type === "waiting for stack sentinel override") {
                if (update.type === "waiting for stack approval") {
                    this.handleUserUpdate(update, {
                        approve: (update) => update.approve(),
                        dismiss: (update) => {
                            update.reject();
                            this.stopAllStacksThatCanNotRunWithout(update.stackName);
                        },
                        stop: (update) => {
                            update.reject();
                            this.stopAllStacks();
                        },
                    }, cb, "waiting for approval");
                }
                else if (update.type === "waiting for stack sentinel override") {
                    this.handleUserUpdate(update, {
                        override: (update) => {
                            update.override();
                        },
                        reject: (update) => {
                            update.reject();
                            this.stopAllStacksThatCanNotRunWithout(update.stackName);
                        },
                    }, cb, "waiting for sentinel override");
                }
                this.ioHandler.awaitUserInput();
            }
            else {
                if (this.ioHandler.isWaitingForUserInput()) {
                    this.ioHandler.pushEvent({
                        cb,
                        value: update,
                        type: "projectUpdate",
                    });
                }
                else {
                    cb(update);
                }
            }
        };
    }
    getStackExecutor(stack, opts = {}) {
        const enhanceLogMessage = (0, execution_logs_1.createEnhanceLogMessage)(stack);
        const onLog = this.ioHandler.bufferWhileAwaitingUserInput(this.onLog);
        return new cdktf_stack_1.CdktfStack({
            ...opts,
            stack,
            onUpdate: this.handleUserInputProcess(this.onUpdate),
            onLog: onLog
                ? ({ message }) => onLog({
                    stackName: stack.name,
                    message,
                    messageWithConstructPath: enhanceLogMessage(message),
                })
                : undefined,
            abortSignal: this.abortSignal,
        });
    }
    get outputsByConstructId() {
        return this.stacksToRun.reduce((acc, stack) => ({
            ...acc,
            ...stack.outputsByConstructId,
        }), {});
    }
    async synth(noColor) {
        this.onUpdate({
            type: "synthesizing",
        });
        const stacks = await synth_stack_1.SynthStack.synth(this.abortSignal, this.synthCommand, this.outDir, this.workingDirectory, false, noColor, this.synthOrigin, this.hcl);
        (0, synth_1.printAnnotations)(stacks);
        this.onUpdate({
            type: "synthesized",
            stacks,
        });
        return stacks;
    }
    async readSynthesizedStacks() {
        const stacks = await synth_stack_1.SynthStack.readSynthesizedStacks(this.outDir);
        (0, synth_1.printAnnotations)(stacks);
        this.onUpdate({
            type: "synthesized",
            stacks,
        });
        return stacks;
    }
    async diff(opts = {}) {
        const stacks = opts.skipSynth
            ? await this.readSynthesizedStacks()
            : await this.synth(opts.noColor);
        const stack = this.getStackExecutor((0, stack_helpers_1.getSingleStack)(stacks, opts === null || opts === void 0 ? void 0 : opts.stackName, "diff"));
        await stack.initalizeTerraform(opts.noColor, opts.skipProviderLock, opts.migrateState);
        try {
            await stack.diff(opts);
        }
        catch (e) {
            throw commons_1.Errors.External(`Stack failed to plan: ${stack.stack.name}. Please check the logs for more information.`, e);
        }
        if (stack.error) {
            throw commons_1.Errors.External(`Stack failed to plan: ${stack.stack.name}. Please check the logs for more information.`, new Error(stack.error));
        }
        try {
            await this.projectTelemetry("diff", {
                stackMetadata: stacks.map((stack) => JSON.parse(stack.content)["//"]
                    ? JSON.parse(stack.content)["//"].metadata
                    : {}),
                errors: stack.error,
                requiredProviders: stacks.map((stack) => JSON.parse(stack.content)["terraform"]
                    ? JSON.parse(stack.content)["terraform"].required_providers
                    : {}),
            });
        }
        catch (e) {
            commons_1.logger.debug("Failed to send telemetry", e);
        }
    }
    async execute(method, next, opts) {
        // We only support refresh only on deploy, a bit of a leaky abstraction here
        if (opts.refreshOnly && method !== "deploy") {
            throw commons_1.Errors.Internal(`Refresh only is only supported on deploy`);
        }
        const maxParallelRuns = !opts.parallelism || opts.parallelism < 0 ? Infinity : opts.parallelism;
        const allExecutions = [];
        await this.initializeStacksToRunInSerial(opts.noColor, opts.skipProviderLock, opts.migrateState);
        while (this.stacksToRun.filter((stack) => stack.isPending).length > 0) {
            const runningStacks = this.stacksToRun.filter((stack) => stack.isRunning);
            if (runningStacks.length >= maxParallelRuns) {
                await Promise.race(runningStacks.map((s) => s.currentWorkPromise));
                continue;
            }
            try {
                const nextRunningExecutor = await next();
                if (!nextRunningExecutor) {
                    // In this case we have no pending stacks, but we also can not find a new executor
                    break;
                }
                const promise = method === "deploy"
                    ? nextRunningExecutor.deploy(opts)
                    : nextRunningExecutor.destroy(opts);
                allExecutions.push(promise);
            }
            catch (e) {
                // await next() threw an error because a stack failed to apply/destroy
                // wait for all other currently running stacks to complete before propagating that error
                commons_1.logger.debug("Encountered an error while awaiting stack to finish", e);
                const openStacks = this.stacksToRun.filter((ex) => ex.currentWorkPromise);
                commons_1.logger.debug("Waiting for still running stacks to finish:", openStacks);
                await Promise.allSettled(openStacks.map((ex) => ex.currentWorkPromise));
                commons_1.logger.debug("Done waiting for still running stacks. All pending work finished");
                throw e;
            }
        }
        // We end the loop when all stacks are started, now we need to wait for them to be done
        // We wait for all work to finish even if one of the promises threw an error.
        await (0, commons_1.ensureAllSettledBeforeThrowing)(Promise.all(allExecutions), allExecutions);
    }
    async deploy(opts = {}) {
        const stacks = opts.skipSynth
            ? await this.readSynthesizedStacks()
            : await this.synth(opts.noColor);
        const stacksToRun = (0, stack_helpers_1.getMultipleStacks)(stacks, opts.stackNames, "deploy");
        if (!opts.ignoreMissingStackDependencies) {
            (0, stack_helpers_1.checkIfAllDependenciesAreIncluded)(stacksToRun);
        }
        this.stopAllStacksThatCanNotRunWithout = (stackName) => {
            (0, stack_helpers_1.findAllNestedDependantStacks)(this.stacksToRun, stackName).forEach((stack) => stack.stop());
        };
        this.stacksToRun = stacksToRun.map((stack) => this.getStackExecutor(stack, opts));
        const next = opts.ignoreMissingStackDependencies
            ? () => Promise.resolve(this.stacksToRun.filter((stack) => stack.isPending)[0])
            : () => (0, stack_helpers_1.getStackWithNoUnmetDependencies)(this.stacksToRun);
        await this.execute("deploy", next, opts);
        const unprocessedStacks = this.stacksToRun.filter((executor) => executor.isPending);
        if (unprocessedStacks.length > 0) {
            throw commons_1.Errors.External(`Some stacks failed to deploy: ${unprocessedStacks
                .map((s) => s.stack.name)
                .join(", ")}. Please check the logs for more information.`);
        }
        try {
            await this.projectTelemetry("deploy", {
                stackMetadata: stacksToRun.map((stack) => JSON.parse(stack.content)["//"]
                    ? JSON.parse(stack.content)["//"].metadata
                    : {}),
                failedStacks: unprocessedStacks.map((stack) => stack.error),
                requiredProviders: stacksToRun.map((stack) => JSON.parse(stack.content)["terraform"]
                    ? JSON.parse(stack.content)["terraform"].required_providers
                    : {}),
            });
        }
        catch (e) {
            commons_1.logger.debug("Failed to send telemetry", e);
        }
    }
    async destroy(opts = {}) {
        const stacks = opts.skipSynth
            ? await this.readSynthesizedStacks()
            : await this.synth(opts.noColor);
        const stacksToRun = (0, stack_helpers_1.getMultipleStacks)(stacks, opts.stackNames, "destroy");
        if (!opts.ignoreMissingStackDependencies) {
            (0, stack_helpers_1.checkIfAllDependantsAreIncluded)(stacksToRun, stacks);
        }
        this.stopAllStacksThatCanNotRunWithout = (stackName) => {
            const stackExecutor = this.stacksToRun.find((s) => s.stack.name === stackName);
            if (!stackExecutor) {
                throw commons_1.Errors.Internal(`Could not find stack "${stackName}" that was stopped`);
            }
            stackExecutor.stack.dependencies.forEach((dependant) => {
                this.stopAllStacksThatCanNotRunWithout(dependant);
                const dependantStack = this.stacksToRun.find((s) => s.stack.name === dependant);
                if (!dependantStack) {
                    throw commons_1.Errors.Internal(`Could not find stack "${dependant}" that was stopped`);
                }
                dependantStack.stop();
            });
        };
        this.stacksToRun = stacksToRun.map((stack) => this.getStackExecutor(stack, opts));
        const next = opts.ignoreMissingStackDependencies
            ? () => Promise.resolve(this.stacksToRun.filter((stack) => stack.currentState !== "done")[0])
            : () => (0, stack_helpers_1.getStackWithNoUnmetDependants)(this.stacksToRun);
        await this.execute("destroy", next, opts);
        const unprocessedStacks = this.stacksToRun.filter((executor) => executor.isPending);
        if (unprocessedStacks.length > 0) {
            throw commons_1.Errors.External(`Some stacks failed to destroy: ${unprocessedStacks
                .map((s) => s.stack.name)
                .join(", ")}. Please check the logs for more information.`);
        }
        try {
            await this.projectTelemetry("destroy", {
                stackMetadata: stacksToRun.map((stack) => JSON.parse(stack.content)["//"]
                    ? JSON.parse(stack.content)["//"].metadata
                    : {}),
                failedStacks: unprocessedStacks.map((stack) => stack.error),
                requiredProviders: stacksToRun.map((stack) => JSON.parse(stack.content)["terraform"]
                    ? JSON.parse(stack.content)["terraform"].required_providers
                    : {}),
            });
        }
        catch (e) {
            commons_1.logger.debug("Failed to send telemetry", e);
        }
    }
    async projectTelemetry(command, payload) {
        const config = (0, commons_1.readConfigSync)();
        await (0, commons_1.sendTelemetry)(command, {
            ...payload,
            language: config.language,
        });
    }
    async fetchOutputs(opts) {
        const stacks = opts.skipSynth
            ? await this.readSynthesizedStacks()
            : await this.synth();
        const stacksToRun = (0, stack_helpers_1.getMultipleStacks)(stacks, opts.stackNames || [], "deploy");
        if (stacksToRun.length === 0) {
            throw new Error("No stacks to fetch outputs for specified");
        }
        this.stacksToRun = stacksToRun.map((stack) => 
        // Options are empty, because MultipleStackOptions doesn't have any relevant
        // options for `getStackExecutor`, hence defaults are fine
        this.getStackExecutor(stack, {}));
        await this.initializeStacksToRunInSerial(undefined, opts.skipProviderLock);
        const outputs = await Promise.all(this.stacksToRun.map(async (s) => {
            const output = await s.fetchOutputs();
            return {
                [s.stack.name]: output,
            };
        }));
        return outputs.reduce((acc, curr) => ({ ...acc, ...curr }), {});
    }
    // Serially run terraform init to prohibit text file busy errors for the cache files
    async initializeStacksToRunInSerial(noColor, skipProviderLock, migrateState) {
        for (const stack of this.stacksToRun) {
            await stack.initalizeTerraform(noColor, skipProviderLock, migrateState);
        }
    }
}
exports.CdktfProject = CdktfProject;
//# sourceMappingURL=data:application/json;base64,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