"use strict";
/**
 * Copyright (c) HashiCorp, Inc.
 * SPDX-License-Identifier: MPL-2.0
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.CdktfProjectIOHandler = void 0;
const cdktf_project_1 = require("./cdktf-project");
/**
 * This class is used to buffer events while the user is being asked for input. It is solely consumed by the CdktfProject class,
 * and is mostly created as a separation of concerns to keep the CdktfProject class as clean as possible.
 */
class CdktfProjectIOHandler {
    constructor() {
        // Pauses all progress / status events from being forwarded to the user
        // If set from true to false, the events will be sent through the channels they came in
        // (until a waiting for approval event is sent)
        this.waitingForUserInput = false;
        this.eventBuffer = [];
    }
    awaitUserInput() {
        this.waitingForUserInput = true;
    }
    filterUserInputEventsFromBuffer() {
        this.eventBuffer = this.eventBuffer.filter((event) => event.type === "projectUpdate"
            ? !(0, cdktf_project_1.isWaitingForUserInputUpdate)(event.value) // we want to filter out the waiting for approval events
            : true);
    }
    resumeAfterUserInput(stackName) {
        // remove waiting for approval event that should be resumed
        this.eventBuffer = this.eventBuffer.filter((event) => !(event.type === "projectUpdate" &&
            (0, cdktf_project_1.isWaitingForUserInputUpdate)(event.value) &&
            event.value.stackName === stackName));
        if (this.eventBuffer.length &&
            this.eventBuffer[0].type === "projectUpdate" &&
            (0, cdktf_project_1.isWaitingForUserInputUpdate)(this.eventBuffer[0].value)) {
            // we are still waiting on approval for the current stack
            // we removed a future "waiting for approval" event for some other stack
            return;
        }
        // We first need to flush all events, we can not resume if there is a new waiting for approval update
        let event = this.eventBuffer.shift();
        while (event) {
            if (event.type === "projectUpdate") {
                event.cb(event.value);
                if ((0, cdktf_project_1.isWaitingForUserInputUpdate)(event.value)) {
                    // We have to wait for user input again, therefore we can not resume
                    return;
                }
            }
            if (event.type === "logMessage") {
                event.cb(event.value);
            }
            event = this.eventBuffer.shift();
        }
        // If we reach this point there was no waiting for user input event, so we can safely resume
        this.waitingForUserInput = false;
    }
    isWaitingForUserInput() {
        return this.waitingForUserInput;
    }
    pushEvent(event) {
        this.eventBuffer.push(event);
    }
    bufferWhileAwaitingUserInput(cb) {
        if (!cb) {
            return undefined;
        }
        return (msg) => {
            if (this.waitingForUserInput) {
                this.eventBuffer.push({
                    cb,
                    value: msg,
                    type: "logMessage",
                });
            }
            else {
                cb(msg);
            }
        };
    }
}
exports.CdktfProjectIOHandler = CdktfProjectIOHandler;
//# sourceMappingURL=data:application/json;base64,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